﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 * <p>Contains the parameters for UnassignPrivateIpAddresses.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/UnassignPrivateIpAddressesRequest">AWS
 * API Reference</a></p>
 */
class UnassignPrivateIpAddressesRequest : public EC2Request {
 public:
  AWS_EC2_API UnassignPrivateIpAddressesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UnassignPrivateIpAddresses"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The IPv4 prefixes to unassign from the network interface.</p>
   */
  inline const Aws::Vector<Aws::String>& GetIpv4Prefixes() const { return m_ipv4Prefixes; }
  inline bool Ipv4PrefixesHasBeenSet() const { return m_ipv4PrefixesHasBeenSet; }
  template <typename Ipv4PrefixesT = Aws::Vector<Aws::String>>
  void SetIpv4Prefixes(Ipv4PrefixesT&& value) {
    m_ipv4PrefixesHasBeenSet = true;
    m_ipv4Prefixes = std::forward<Ipv4PrefixesT>(value);
  }
  template <typename Ipv4PrefixesT = Aws::Vector<Aws::String>>
  UnassignPrivateIpAddressesRequest& WithIpv4Prefixes(Ipv4PrefixesT&& value) {
    SetIpv4Prefixes(std::forward<Ipv4PrefixesT>(value));
    return *this;
  }
  template <typename Ipv4PrefixesT = Aws::String>
  UnassignPrivateIpAddressesRequest& AddIpv4Prefixes(Ipv4PrefixesT&& value) {
    m_ipv4PrefixesHasBeenSet = true;
    m_ipv4Prefixes.emplace_back(std::forward<Ipv4PrefixesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the network interface.</p>
   */
  inline const Aws::String& GetNetworkInterfaceId() const { return m_networkInterfaceId; }
  inline bool NetworkInterfaceIdHasBeenSet() const { return m_networkInterfaceIdHasBeenSet; }
  template <typename NetworkInterfaceIdT = Aws::String>
  void SetNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    m_networkInterfaceIdHasBeenSet = true;
    m_networkInterfaceId = std::forward<NetworkInterfaceIdT>(value);
  }
  template <typename NetworkInterfaceIdT = Aws::String>
  UnassignPrivateIpAddressesRequest& WithNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    SetNetworkInterfaceId(std::forward<NetworkInterfaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The secondary private IP addresses to unassign from the network interface.
   * You can specify this option multiple times to unassign more than one IP
   * address.</p>
   */
  inline const Aws::Vector<Aws::String>& GetPrivateIpAddresses() const { return m_privateIpAddresses; }
  inline bool PrivateIpAddressesHasBeenSet() const { return m_privateIpAddressesHasBeenSet; }
  template <typename PrivateIpAddressesT = Aws::Vector<Aws::String>>
  void SetPrivateIpAddresses(PrivateIpAddressesT&& value) {
    m_privateIpAddressesHasBeenSet = true;
    m_privateIpAddresses = std::forward<PrivateIpAddressesT>(value);
  }
  template <typename PrivateIpAddressesT = Aws::Vector<Aws::String>>
  UnassignPrivateIpAddressesRequest& WithPrivateIpAddresses(PrivateIpAddressesT&& value) {
    SetPrivateIpAddresses(std::forward<PrivateIpAddressesT>(value));
    return *this;
  }
  template <typename PrivateIpAddressesT = Aws::String>
  UnassignPrivateIpAddressesRequest& AddPrivateIpAddresses(PrivateIpAddressesT&& value) {
    m_privateIpAddressesHasBeenSet = true;
    m_privateIpAddresses.emplace_back(std::forward<PrivateIpAddressesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_ipv4Prefixes;
  bool m_ipv4PrefixesHasBeenSet = false;

  Aws::String m_networkInterfaceId;
  bool m_networkInterfaceIdHasBeenSet = false;

  Aws::Vector<Aws::String> m_privateIpAddresses;
  bool m_privateIpAddressesHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
