package format

import "github.com/wader/fq/pkg/scalar"

// from https://www.tcpdump.org/linktypes.html
// TODO cleanup
//
//nolint:revive
const (
	LinkTypeNULL                       = 0
	LinkTypeETHERNET                   = 1
	LinkTypeAX25                       = 3
	LinkTypeIEEE802_5                  = 6
	LinkTypeARCNET_BSD                 = 7
	LinkTypeSLIP                       = 8
	LinkTypePPP                        = 9
	LinkTypeFDDI                       = 10
	LinkTypePPP_HDLC                   = 50
	LinkTypePPP_ETHER                  = 51
	LinkTypeATM_RFC1483                = 100
	LinkTypeRAW                        = 101
	LinkTypeC_HDLC                     = 104
	LinkTypeIEEE802_11                 = 105
	LinkTypeFRELAY                     = 107
	LinkTypeLOOP                       = 108
	LinkTypeLINUX_SLL                  = 113
	LinkTypeLTALK                      = 114
	LinkTypePFLOG                      = 117
	LinkTypeIEEE802_11_PRISM           = 119
	LinkTypeIP_OVER_FC                 = 122
	LinkTypeSUNATM                     = 123
	LinkTypeIEEE802_11_RADIOTAP        = 127
	LinkTypeARCNET_LINUX               = 129
	LinkTypeAPPLE_IP_OVER_IEEE1394     = 138
	LinkTypeMTP2_WITH_PHDR             = 139
	LinkTypeMTP2                       = 140
	LinkTypeMTP3                       = 141
	LinkTypeSCCP                       = 142
	LinkTypeDOCSIS                     = 143
	LinkTypeLINUX_IRDA                 = 144
	LinkTypeUSER0                      = 147
	LinkTypeUSER1                      = 148
	LinkTypeUSER2                      = 149
	LinkTypeUSER3                      = 150
	LinkTypeUSER4                      = 151
	LinkTypeUSER5                      = 152
	LinkTypeUSER6                      = 153
	LinkTypeUSER7                      = 154
	LinkTypeUSER8                      = 155
	LinkTypeUSER9                      = 156
	LinkTypeUSER10                     = 157
	LinkTypeUSER11                     = 158
	LinkTypeUSER12                     = 159
	LinkTypeUSER13                     = 160
	LinkTypeUSER14                     = 161
	LinkTypeUSER15                     = 162
	LinkTypeIEEE802_11_AVS             = 163
	LinkTypeBACNET_MS_TP               = 165
	LinkTypePPP_PPPD                   = 166
	LinkTypeGPRS_LLC                   = 169
	LinkTypeGPF_T                      = 170
	LinkTypeGPF_F                      = 171
	LinkTypeLINUX_LAPD                 = 177
	LinkTypeMFR                        = 182
	LinkTypeBLUETOOTH_HCI_H4           = 187
	LinkTypeUSB_LINUX                  = 189
	LinkTypePPI                        = 192
	LinkTypeIEEE802_15_4_WITHFCS       = 195
	LinkTypeSITA                       = 196
	LinkTypeERF                        = 197
	LinkTypeBLUETOOTH_HCI_H4_WITH_PHDR = 201
	LinkTypeAX25_KISS                  = 202
	LinkTypeLAPD                       = 203
	LinkTypePPP_WITH_DIR               = 204
	LinkTypeC_HDLC_WITH_DIR            = 205
	LinkTypeFRELAY_WITH_DIR            = 206
	LinkTypeLAPB_WITH_DIR              = 207
	LinkTypeIPMB_LINUX                 = 209
	LinkTypeFLEXRAY                    = 210
	LinkTypeLIN                        = 212
	LinkTypeIEEE802_15_4_NONASK_PHY    = 215
	LinkTypeUSB_LINUX_MMAPPED          = 220
	LinkTypeFC_2                       = 224
	LinkTypeFC_2_WITH_FRAME_DELIMS     = 225
	LinkTypeIPNET                      = 226
	LinkTypeCAN_SOCKETCAN              = 227
	LinkTypeIPV4                       = 228
	LinkTypeIPV6                       = 229
	LinkTypeIEEE802_15_4_NOFCS         = 230
	LinkTypeDBUS                       = 231
	LinkTypeDVB_CI                     = 235
	LinkTypeMUX27010                   = 236
	LinkTypeSTANAG_5066_D_PDU          = 237
	LinkTypeNFLOG                      = 239
	LinkTypeNETANALYZER                = 240
	LinkTypeNETANALYZER_TRANSPARENT    = 241
	LinkTypeIPOIB                      = 242
	LinkTypeMPEG_2_TS                  = 243
	LinkTypeNG40                       = 244
	LinkTypeNFC_LLCP                   = 245
	LinkTypeINFINIBAND                 = 247
	LinkTypeSCTP                       = 248
	LinkTypeUSBPCAP                    = 249
	LinkTypeRTAC_SERIAL                = 250
	LinkTypeBLUETOOTH_LE_LL            = 251
	LinkTypeNETLINK                    = 253
	LinkTypeBLUETOOTH_LINUX_MONITOR    = 254
	LinkTypeBLUETOOTH_BREDR_BB         = 255
	LinkTypeBLUETOOTH_LE_LL_WITH_PHDR  = 256
	LinkTypePROFIBUS_DL                = 257
	LinkTypePKTAP                      = 258
	LinkTypeEPON                       = 259
	LinkTypeIPMI_HPM_2                 = 260
	LinkTypeZWAVE_R1_R2                = 261
	LinkTypeZWAVE_R3                   = 262
	LinkTypeWATTSTOPPER_DLM            = 263
	LinkTypeISO_14443                  = 264
	LinkTypeRDS                        = 265
	LinkTypeUSB_DARWIN                 = 266
	LinkTypeSDLC                       = 268
	LinkTypeLORATAP                    = 270
	LinkTypeVSOCK                      = 271
	LinkTypeNORDIC_BLE                 = 272
	LinkTypeDOCSIS31_XRA31             = 273
	LinkTypeETHERNET_MPACKET           = 274
	LinkTypeDISPLAYPORT_AUX            = 275
	LinkTypeLINUX_SLL2                 = 276
	LinkTypeOPENVIZSLA                 = 278
	LinkTypeEBHSCR                     = 279
	LinkTypeVPP_DISPATCH               = 280
	LinkTypeDSA_TAG_BRCM               = 281
	LinkTypeDSA_TAG_BRCM_PREPEND       = 282
	LinkTypeIEEE802_15_4_TAP           = 283
	LinkTypeDSA_TAG_DSA                = 284
	LinkTypeDSA_TAG_EDSA               = 285
	LinkTypeELEE                       = 286
	LinkTypeZ_WAVE_SERIAL              = 287
	LinkTypeUSB_2_0                    = 288
	LinkTypeATSC_ALP                   = 289
	LinkTypeETW                        = 290
)

var LinkTypeMap = scalar.UintMap{
	LinkTypeNULL:                       {Sym: "null", Description: `BSD loopback encapsulation`},
	LinkTypeETHERNET:                   {Sym: "ethernet", Description: `IEEE 802.3 Ethernet`},
	LinkTypeAX25:                       {Sym: "ax25", Description: `AX.25 packet, with nothing preceding it`},
	LinkTypeIEEE802_5:                  {Sym: "ieee802_5", Description: `IEEE 802.5 Token Ring`},
	LinkTypeARCNET_BSD:                 {Sym: "arcnet_bsd", Description: `ARCNET Data Packets`},
	LinkTypeSLIP:                       {Sym: "slip", Description: `SLIP, encapsulated with a LINKTYPE_SLIP header`},
	LinkTypePPP:                        {Sym: "ppp", Description: `PPP`},
	LinkTypeFDDI:                       {Sym: "fddi", Description: `FDDI`},
	LinkTypePPP_HDLC:                   {Sym: "ppp_hdlc", Description: `PPP in HDLC-like framing`},
	LinkTypePPP_ETHER:                  {Sym: "ppp_ether", Description: `PPPoE`},
	LinkTypeATM_RFC1483:                {Sym: "atm_rfc1483", Description: `RFC 1483 LLC/SNAP-encapsulated ATM`},
	LinkTypeRAW:                        {Sym: "raw", Description: `Raw IP`},
	LinkTypeC_HDLC:                     {Sym: "c_hdlc", Description: `Cisco PPP with HDLC framing, as per section 4.3.1 of RFC 1547`},
	LinkTypeIEEE802_11:                 {Sym: "ieee802_11", Description: `IEEE 802.11 wireless LAN`},
	LinkTypeFRELAY:                     {Sym: "frelay", Description: `Frame Relay LAPF frames`},
	LinkTypeLOOP:                       {Sym: "loop", Description: `OpenBSD loopback encapsulation`},
	LinkTypeLINUX_SLL:                  {Sym: "linux_sll", Description: `Linux "cooked" capture encapsulation`},
	LinkTypeLTALK:                      {Sym: "ltalk", Description: `Apple LocalTalk`},
	LinkTypePFLOG:                      {Sym: "pflog", Description: `OpenBSD pflog`},
	LinkTypeIEEE802_11_PRISM:           {Sym: "ieee802_11_prism", Description: `Prism monitor mode information followed by an 802.11 header`},
	LinkTypeIP_OVER_FC:                 {Sym: "ip_over_fc", Description: `RFC 2625 IP-over-Fibre Channel, with the link-layer header being the Network_Header as described in that RFC`},
	LinkTypeSUNATM:                     {Sym: "sunatm", Description: `ATM traffic, encapsulated as per the scheme used by SunATM devices`},
	LinkTypeIEEE802_11_RADIOTAP:        {Sym: "ieee802_11_radiotap", Description: `Radiotap link-layer information followed by an 802.11 header`},
	LinkTypeARCNET_LINUX:               {Sym: "arcnet_linux", Description: `ARCNET Data Packets`},
	LinkTypeAPPLE_IP_OVER_IEEE1394:     {Sym: "apple_ip_over_ieee1394", Description: `Apple IP-over-IEEE 1394 cooked header`},
	LinkTypeMTP2_WITH_PHDR:             {Sym: "mtp2_with_phdr", Description: `Signaling System 7 Message Transfer Part Level 2`},
	LinkTypeMTP2:                       {Sym: "mtp2", Description: `Signaling System 7 Message Transfer Part Level 2`},
	LinkTypeMTP3:                       {Sym: "mtp3", Description: `Signaling System 7 Message Transfer Part Level 3`},
	LinkTypeSCCP:                       {Sym: "sccp", Description: `Signaling System 7 Signalling Connection Control Part`},
	LinkTypeDOCSIS:                     {Sym: "docsis", Description: `DOCSIS MAC frames`},
	LinkTypeLINUX_IRDA:                 {Sym: "linux_irda", Description: `Linux-IrDA packets, with a LINKTYPE_LINUX_IRDA header`},
	LinkTypeUSER0:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER1:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER2:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER3:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER4:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER5:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER6:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER7:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER8:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER9:                      {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER10:                     {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER11:                     {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER12:                     {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER13:                     {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER14:                     {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeUSER15:                     {Sym: "user0", Description: `Reserved for private use`},
	LinkTypeIEEE802_11_AVS:             {Sym: "ieee802_11_avs", Description: `AVS monitor mode information followed by an 802.11 header`},
	LinkTypeBACNET_MS_TP:               {Sym: "bacnet_ms_tp", Description: `BACnet MS/TP frames`},
	LinkTypePPP_PPPD:                   {Sym: "ppp_pppd", Description: `PPP in HDLC-like encapsulation, like LINKTYPE_PPP_HDLC`},
	LinkTypeGPRS_LLC:                   {Sym: "gprs_llc", Description: `General Packet Radio Service Logical Link Control, as defined by 3GPP TS 04.64`},
	LinkTypeGPF_T:                      {Sym: "gpf_t", Description: `Transparent-mapped generic framing procedure`},
	LinkTypeGPF_F:                      {Sym: "gpf_f", Description: `Frame-mapped generic framing procedure`},
	LinkTypeLINUX_LAPD:                 {Sym: "linux_lapd", Description: `Link Access Procedures on the D Channel (LAPD) frames`},
	LinkTypeMFR:                        {Sym: "mfr", Description: `FRF.16.1 Multi-Link Frame Relay frames, beginning with an FRF.12 Interface fragmentation format fragmentation header`},
	LinkTypeBLUETOOTH_HCI_H4:           {Sym: "bluetooth_hci_h4", Description: `Bluetooth HCI UART transport layer`},
	LinkTypeUSB_LINUX:                  {Sym: "usb_linux", Description: `USB packets, beginning with a Linux USB header`},
	LinkTypePPI:                        {Sym: "ppi", Description: `Per-Packet Information information`},
	LinkTypeIEEE802_15_4_WITHFCS:       {Sym: "ieee802_15_4_withfcs", Description: `IEEE 802.15.4 Low-Rate Wireless Networks, with each packet having the FCS at the end of the frame`},
	LinkTypeSITA:                       {Sym: "sita", Description: `Various link-layer types, with a pseudo-header, for SITA`},
	LinkTypeERF:                        {Sym: "erf", Description: `Various link-layer types, with a pseudo-header, for Endace DAG cards`},
	LinkTypeBLUETOOTH_HCI_H4_WITH_PHDR: {Sym: "bluetooth_hci_h4_with_phdr", Description: `Bluetooth HCI UART transport layer`},
	LinkTypeAX25_KISS:                  {Sym: "ax25_kiss", Description: `AX.25 packet, with a 1-byte KISS header containing a type indicator`},
	LinkTypeLAPD:                       {Sym: "lapd", Description: `Link Access Procedures on the D Channel (LAPD) frames`},
	LinkTypePPP_WITH_DIR:               {Sym: "ppp_with_dir", Description: `PPP, as per RFC 1661 and RFC 1662`},
	LinkTypeC_HDLC_WITH_DIR:            {Sym: "c_hdlc_with_dir", Description: `Cisco PPP with HDLC framing`},
	LinkTypeFRELAY_WITH_DIR:            {Sym: "frelay_with_dir", Description: `Frame Relay LAPF frames`},
	LinkTypeLAPB_WITH_DIR:              {Sym: "lapb_with_dir", Description: `Link Access Procedure, Balanced (LAPB)`},
	LinkTypeIPMB_LINUX:                 {Sym: "ipmb_linux", Description: `IPMB over an I2C circuit, with a Linux-specific pseudo-header`},
	LinkTypeFLEXRAY:                    {Sym: "flexray", Description: `FlexRay automotive bus frames or symbols, preceded by a pseudo-header`},
	LinkTypeLIN:                        {Sym: "lin", Description: `Local Interconnect Network (LIN) automotive bus, preceded by a pseudo-header`},
	LinkTypeIEEE802_15_4_NONASK_PHY:    {Sym: "ieee802_15_4_nonask_phy", Description: `IEEE 802.15.4 Low-Rate Wireless Networks`},
	LinkTypeUSB_LINUX_MMAPPED:          {Sym: "usb_linux_mmapped", Description: `USB packets, beginning with a Linux USB header`},
	LinkTypeFC_2:                       {Sym: "fc_2", Description: `Fibre Channel FC-2 frames, beginning with a Frame_Header`},
	LinkTypeFC_2_WITH_FRAME_DELIMS:     {Sym: "fc_2_with_frame_delims", Description: `Fibre Channel FC-2 frames, beginning an encoding of the SOF, followed by a Frame_Header, and ending with an encoding of the SOF`},
	LinkTypeIPNET:                      {Sym: "ipnet", Description: `Solaris ipnet pseudo-header, followed by an IPv4 or IPv6 datagram`},
	LinkTypeCAN_SOCKETCAN:              {Sym: "can_socketcan", Description: `CAN (Controller Area Network) frames, with a pseudo-header followed by the frame payload`},
	LinkTypeIPV4:                       {Sym: "ipv4", Description: `Raw IPv4`},
	LinkTypeIPV6:                       {Sym: "ipv6", Description: `Raw IPv6`},
	LinkTypeIEEE802_15_4_NOFCS:         {Sym: "ieee802_15_4_nofcs", Description: `IEEE 802.15.4 Low-Rate Wireless Network, without the FCS at the end of the frame`},
	LinkTypeDBUS:                       {Sym: "dbus", Description: `Raw D-Bus messages`},
	LinkTypeDVB_CI:                     {Sym: "dvb_ci", Description: `DVB-CI (DVB Common Interface for communication between a PC Card module and a DVB receiver)`},
	LinkTypeMUX27010:                   {Sym: "mux27010", Description: `Variant of 3GPP TS 27.010 multiplexing protocol (similar to, but not the same as, 27.010)`},
	LinkTypeSTANAG_5066_D_PDU:          {Sym: "stanag_5066_d_pdu", Description: `D_PDUs as described by NATO standard STANAG 5066`},
	LinkTypeNFLOG:                      {Sym: "nflog", Description: `Linux netlink NETLINK NFLOG socket log messages`},
	LinkTypeNETANALYZER:                {Sym: "netanalyzer", Description: `Pseudo-header for Hilscher Gesellschaft für Systemautomation mbH netANALYZER devices`},
	LinkTypeNETANALYZER_TRANSPARENT:    {Sym: "netanalyzer_transparent", Description: `Pseudo-header for Hilscher Gesellschaft für Systemautomation mbH netANALYZER devices`},
	LinkTypeIPOIB:                      {Sym: "ipoib", Description: `IP-over-InfiniBand`},
	LinkTypeMPEG_2_TS:                  {Sym: "mpeg_2_ts", Description: `MPEG-2 Transport Stream transport packets`},
	LinkTypeNG40:                       {Sym: "ng40", Description: `Pseudo-header for ng4T GmbH's UMTS Iub/Iur-over-ATM and Iub/Iur-over-IP forma`},
	LinkTypeNFC_LLCP:                   {Sym: "nfc_llcp", Description: `Pseudo-header for NFC LLCP packet captures, followed by frame data for the LLCP Protocol as specified by NFCForum-TS-LLCP_1.1`},
	LinkTypeINFINIBAND:                 {Sym: "infiniband", Description: `Raw InfiniBand frames`},
	LinkTypeSCTP:                       {Sym: "sctp", Description: `SCTP packets, as defined by RFC 4960, with no lower-level protocols such as IPv4 or IPv6`},
	LinkTypeUSBPCAP:                    {Sym: "usbpcap", Description: `USB packets, beginning with a USBPcap header`},
	LinkTypeRTAC_SERIAL:                {Sym: "rtac_serial", Description: `Serial-line packet header for the Schweitzer Engineering Laboratories "RTAC" product`},
	LinkTypeBLUETOOTH_LE_LL:            {Sym: "bluetooth_le_ll", Description: `Bluetooth Low Energy air interface Link Layer packets`},
	LinkTypeNETLINK:                    {Sym: "netlink", Description: `Linux Netlink capture encapsulation`},
	LinkTypeBLUETOOTH_LINUX_MONITOR:    {Sym: "bluetooth_linux_monitor", Description: `Bluetooth Linux Monitor encapsulation of traffic for the BlueZ stack`},
	LinkTypeBLUETOOTH_BREDR_BB:         {Sym: "bluetooth_bredr_bb", Description: `Bluetooth Basic Rate and Enhanced Data Rate baseband packets`},
	LinkTypeBLUETOOTH_LE_LL_WITH_PHDR:  {Sym: "bluetooth_le_ll_with_phdr", Description: `Bluetooth Low Energy link-layer packets`},
	LinkTypePROFIBUS_DL:                {Sym: "profibus_dl", Description: `PROFIBUS data link layer packets`},
	LinkTypePKTAP:                      {Sym: "pktap", Description: `Apple PKTAP capture encapsulation`},
	LinkTypeEPON:                       {Sym: "epon", Description: `Ethernet-over-passive-optical-network packets`},
	LinkTypeIPMI_HPM_2:                 {Sym: "ipmi_hpm_2", Description: `IPMI trace packets`},
	LinkTypeZWAVE_R1_R2:                {Sym: "zwave_r1_r2", Description: `Z-Wave RF profile R1 and R2 packets`},
	LinkTypeZWAVE_R3:                   {Sym: "zwave_r3", Description: `Z-Wave RF profile R3 packets`},
	LinkTypeWATTSTOPPER_DLM:            {Sym: "wattstopper_dlm", Description: `Formats for WattStopper Digital Lighting Management (DLM) and Legrand Nitoo Open protocol common packet structure captures`},
	LinkTypeISO_14443:                  {Sym: "iso_14443", Description: `Messages between ISO 14443 contactless smartcards (Proximity Integrated Circuit Card, PICC) and card readers (Proximity Coupling Device, PCD), with the message format specified by the PCAP format for ISO14443 specification`},
	LinkTypeRDS:                        {Sym: "rds", Description: `Radio data system (RDS) groups, as per IEC 62106, encapsulated in this form`},
	LinkTypeUSB_DARWIN:                 {Sym: "usb_darwin", Description: `USB packets, beginning with a Darwin (macOS, etc.) USB header`},
	LinkTypeSDLC:                       {Sym: "sdlc", Description: `SDLC packets`},
	LinkTypeLORATAP:                    {Sym: "loratap", Description: `LoRaTap pseudo-header, followed by the payload, which is typically the PHYPayload from the LoRaWan specification`},
	LinkTypeVSOCK:                      {Sym: "vsock", Description: `Protocol for communication between host and guest machines in VMware and KVM hypervisors`},
	LinkTypeNORDIC_BLE:                 {Sym: "nordic_ble", Description: `Messages to and from a Nordic Semiconductor nRF Sniffer for Bluetooth LE packets, beginning with a pseudo-header`},
	LinkTypeDOCSIS31_XRA31:             {Sym: "docsis31_xra31", Description: `DOCSIS packets and bursts, preceded by a pseudo-header giving metadata about the packet`},
	LinkTypeETHERNET_MPACKET:           {Sym: "ethernet_mpacket", Description: `mPackets`},
	LinkTypeDISPLAYPORT_AUX:            {Sym: "displayport_aux", Description: `DisplayPort AUX channel monitoring data as specified by VESA DisplayPort(DP) Standard preceded by a pseudo-header`},
	LinkTypeLINUX_SLL2:                 {Sym: "linux_sll2", Description: `Linux "cooked" capture encapsulation v2`},
	LinkTypeOPENVIZSLA:                 {Sym: "openvizsla", Description: `Openvizsla FPGA-based USB sniffer`},
	LinkTypeEBHSCR:                     {Sym: "ebhscr", Description: `Elektrobit High Speed Capture and Replay (EBHSCR) format`},
	LinkTypeVPP_DISPATCH:               {Sym: "vpp_dispatch", Description: `Records in traces from the http://fd.io VPP graph dispatch tracer`},
	LinkTypeDSA_TAG_BRCM:               {Sym: "dsa_tag_brcm", Description: `Ethernet frames, with a switch tag inserted between the source address field and the type/length field in the Ethernet header`},
	LinkTypeDSA_TAG_BRCM_PREPEND:       {Sym: "dsa_tag_brcm_prepend", Description: `Ethernet frames, with a switch tag inserted before the destination address in the Ethernet header`},
	LinkTypeIEEE802_15_4_TAP:           {Sym: "ieee802_15_4_tap", Description: `IEEE 802.15.4 Low-Rate Wireless Networks, with a pseudo-header containing TLVs with metadata preceding the 802.15.4 header`},
	LinkTypeDSA_TAG_DSA:                {Sym: "dsa_tag_dsa", Description: `Ethernet frames, with a switch tag inserted between the source address field and the type/length field in the Ethernet header`},
	LinkTypeDSA_TAG_EDSA:               {Sym: "dsa_tag_edsa", Description: `Ethernet frames, with a programmable Ethernet type switch tag`},
	LinkTypeELEE:                       {Sym: "elee", Description: `Payload of lawful intercept packets using the ELEE protocol. The packet begins with the ELEE header`},
	LinkTypeZ_WAVE_SERIAL:              {Sym: "z_wave_serial", Description: `Serial frames transmitted between a host and a Z-Wave chip over an RS-232 or USB serial connection`},
	LinkTypeUSB_2_0:                    {Sym: "usb_2_0", Description: `USB 2.0, 1.1, or 1.0 packet, beginning with a PID`},
	LinkTypeATSC_ALP:                   {Sym: "atsc_alp", Description: `ATSC Link-Layer Protocol frames`},
	LinkTypeETW:                        {Sym: "etw", Description: `Event Tracing for Windows messages, beginning with a pseudo-header`},
}

const (
	EtherTypeIPv4 = 0x0800
	EtherTypeIPv6 = 0x86dd
)

// from https://en.wikipedia.org/wiki/EtherType
// TODO: cleanup
var EtherTypeMap = scalar.UintMap{
	EtherTypeIPv4: {Sym: "ipv4", Description: `Internet Protocol version 4`},
	0x0806:        {Sym: "arp", Description: `Address Resolution Protocol`},
	0x0842:        {Sym: "wake", Description: `Wake-on-LAN[9]`},
	0x22f0:        {Sym: "audio", Description: `Audio Video Transport Protocol`},
	0x22f3:        {Sym: "trill", Description: `IETF TRILL Protocol`},
	0x22ea:        {Sym: "srp", Description: `Stream Reservation Protocol`},
	0x6002:        {Sym: "dec", Description: `DEC MOP RC`},
	0x6003:        {Sym: "decnet", Description: `DECnet Phase IV, DNA Routing`},
	0x6004:        {Sym: "declat", Description: `DEC LAT`},
	0x8035:        {Sym: "reverse", Description: `Reverse Address Resolution Protocol`},
	0x809b:        {Sym: "appletalk", Description: `AppleTalk`},
	0x80f3:        {Sym: "appletalk_arp", Description: `AppleTalk Address Resolution Protocol`},
	0x8100:        {Sym: "vlan", Description: `VLAN-tagged (IEEE 802.1Q)`},
	0x8102:        {Sym: "slpp", Description: `Simple Loop Prevention Protocol`},
	0x8103:        {Sym: "vlacp", Description: `Virtual Link Aggregation Control Protocol`},
	0x8137:        {Sym: "ipx", Description: `IPX`},
	0x8204:        {Sym: "qnx", Description: `QNX Qnet`},
	EtherTypeIPv6: {Sym: "ipv6", Description: `Internet Protocol Version 6`},
	0x8808:        {Sym: "flow_control", Description: `Ethernet flow control`},
	0x8809:        {Sym: "lacp", Description: `Ethernet Slow Protocols] such as the Link Aggregation Control Protocol`},
	0x8819:        {Sym: "cobranet", Description: `CobraNet`},
	0x8847:        {Sym: "mpls", Description: `MPLS unicast`},
	0x8848:        {Sym: "mpls", Description: `MPLS multicast`},
	0x8863:        {Sym: "pppoe_discovery", Description: `PPPoE Discovery Stage`},
	0x8864:        {Sym: "pppoe_session", Description: `PPPoE Session Stage`},
	0x887b:        {Sym: "homeplug", Description: `HomePlug 1.0 MME`},
	0x888e:        {Sym: "eap", Description: `EAP over LAN (IEEE 802.1X)`},
	0x8892:        {Sym: "profinet", Description: `PROFINET Protocol`},
	0x889a:        {Sym: "hyperscsi", Description: `HyperSCSI (SCSI over Ethernet)`},
	0x88a2:        {Sym: "ata", Description: `ATA over Ethernet`},
	0x88a4:        {Sym: "ethercat", Description: `EtherCAT Protocol`},
	0x88a8:        {Sym: "service", Description: `Service VLAN tag identifier (S-Tag) on Q-in-Q tunnel`},
	0x88ab:        {Sym: "ethernet", Description: `Ethernet Powerlink`},
	0x88b8:        {Sym: "goose", Description: `GOOSE (Generic Object Oriented Substation event)`},
	0x88b9:        {Sym: "gse", Description: `GSE (Generic Substation Events) Management Services`},
	0x88ba:        {Sym: "sv", Description: `SV (Sampled Value Transmission)`},
	0x88bf:        {Sym: "mikrotik", Description: `MikroTik RoMON (unofficial)`},
	0x88cc:        {Sym: "link", Description: `Link Layer Discovery Protocol (LLDP)`},
	0x88cd:        {Sym: "sercos", Description: `SERCOS III`},
	0x88e1:        {Sym: "homeplug", Description: `HomePlug Green PHY`},
	0x88e3:        {Sym: "media", Description: `Media Redundancy Protocol (IEC62439-2)`},
	0x88e5:        {Sym: "ieee", Description: `IEEE 802.1AE MAC security (MACsec)`},
	0x88e7:        {Sym: "provider", Description: `Provider Backbone Bridges (PBB) (IEEE 802.1ah)`},
	0x88f7:        {Sym: "precision", Description: `Precision Time Protocol (PTP) over IEEE 802.3 Ethernet`},
	0x88f8:        {Sym: "nc", Description: `NC-SI`},
	0x88fb:        {Sym: "parallel", Description: `Parallel Redundancy Protocol (PRP)`},
	0x8902:        {Sym: "ieee", Description: `IEEE 802.1ag Connectivity Fault Management (CFM) Protocol / ITU-T Recommendation Y.1731 (OAM)`},
	0x8906:        {Sym: "fibre", Description: `Fibre Channel over Ethernet (FCoE)`},
	0x8914:        {Sym: "fcoe", Description: `FCoE Initialization Protocol`},
	0x8915:        {Sym: "rdma", Description: `RDMA over Converged Ethernet (RoCE)`},
	0x891d:        {Sym: "ttethernet", Description: `TTEthernet Protocol Control Frame (TTE)`},
	0x893a:        {Sym: "1905", Description: `1905.1 IEEE Protocol`},
	0x892f:        {Sym: "high", Description: `High-availability Seamless Redundancy (HSR)`},
	0x9000:        {Sym: "ethernet", Description: `Ethernet Configuration Testing Protocol[12]`},
	0xf1c1:        {Sym: "redundancy", Description: `Redundancy Tag (IEEE 802.1CB Frame Replication and Elimination for Reliability)`},
}

// based on etc/protocols from Darwin/FreeBSD
// cat /etc/protocols | grep -v '^#'  | jq -rR 'capture("(?<name>[\\w\\d-]+)\\s+(?<nr>\\d+)\\s+.*#\\s+(?<desc>.*)") | "\(.nr): {Sym: \(.name|tojson), Description: \(.desc|tojson)},"'

const (
	IPv4ProtocolICMP   = 1
	IPv4ProtocolIGMP   = 2
	IPv4ProtocolTCP    = 6
	IPv4ProtocolUDP    = 17
	IPv4ProtocolICMPv6 = 58
)

var IPv4ProtocolMap = scalar.UintMap{
	0:                  {Sym: "ip", Description: "Internet protocol, pseudo protocol number"},
	IPv4ProtocolICMP:   {Sym: "icmp", Description: "Internet control message protocol"},
	IPv4ProtocolIGMP:   {Sym: "igmp", Description: "Internet group management protocol"},
	3:                  {Sym: "ggp", Description: "Gateway-gateway protocol"},
	4:                  {Sym: "ipencap", Description: "IP encapsulated in IP"},
	5:                  {Sym: "st2", Description: "ST2 datagram mode"},
	IPv4ProtocolTCP:    {Sym: "tcp", Description: "Transmission control protocol"},
	7:                  {Sym: "cbt"},
	8:                  {Sym: "egp", Description: "Exterior gateway protocol"},
	9:                  {Sym: "igp", Description: "Any private interior gateway"},
	10:                 {Sym: "bbn-rcc", Description: "BBN RCC Monitoring"},
	11:                 {Sym: "nvp", Description: "Network Voice Protocol"},
	12:                 {Sym: "pup", Description: "PARC universal packet protocol"},
	13:                 {Sym: "argus", Description: "ARGUS"},
	14:                 {Sym: "emcon", Description: "EMCON"},
	15:                 {Sym: "xnet", Description: "Cross Net Debugger"},
	16:                 {Sym: "chaos", Description: "Chaos"},
	IPv4ProtocolUDP:    {Sym: "udp", Description: "User datagram protocol"},
	18:                 {Sym: "mux", Description: "Multiplexing protocol"},
	19:                 {Sym: "dcn", Description: "DCN Measurement Subsystems"},
	20:                 {Sym: "hmp", Description: "Host monitoring protocol"},
	21:                 {Sym: "prm", Description: "Packet radio measurement protocol"},
	22:                 {Sym: "xns-idp", Description: "Xerox NS IDP"},
	23:                 {Sym: "trunk-1", Description: "Trunk-1"},
	24:                 {Sym: "trunk-2", Description: "Trunk-2"},
	25:                 {Sym: "leaf-1", Description: "Leaf-1"},
	26:                 {Sym: "leaf-2", Description: "Leaf-2"},
	27:                 {Sym: "rdp", Description: "Reliable datagram protocol"},
	28:                 {Sym: "irtp", Description: "Internet Reliable Transaction Protocol"},
	29:                 {Sym: "iso-tp4", Description: "ISO Transport Protocol Class 4"},
	30:                 {Sym: "netblt", Description: "Bulk Data Transfer Protocol"},
	31:                 {Sym: "mfe-nsp", Description: "MFE Network Services Protocol"},
	32:                 {Sym: "merit-inp", Description: "MERIT Internodal Protocol"},
	33:                 {Sym: "dccp", Description: "Datagram Congestion Control Protocol"},
	34:                 {Sym: "3pc", Description: "Third Party Connect Protocol"},
	35:                 {Sym: "idpr", Description: "Inter-Domain Policy Routing Protocol"},
	36:                 {Sym: "xtp", Description: "Xpress Tranfer Protocol"},
	37:                 {Sym: "ddp", Description: "Datagram Delivery Protocol"},
	38:                 {Sym: "idpr-cmtp", Description: "IDPR Control Message Transport Proto"},
	40:                 {Sym: "il", Description: "IL Transport Protocol"},
	41:                 {Sym: "ipv6", Description: "IPv6"},
	42:                 {Sym: "sdrp", Description: "Source Demand Routing Protocol"},
	43:                 {Sym: "ipv6-route", Description: "routing header for ipv6"},
	44:                 {Sym: "ipv6-frag", Description: "fragment header for ipv6"},
	45:                 {Sym: "idrp", Description: "Inter-Domain Routing Protocol"},
	46:                 {Sym: "rsvp", Description: "Resource ReSerVation Protocol"},
	47:                 {Sym: "gre", Description: "Generic Routing Encapsulation"},
	48:                 {Sym: "dsr", Description: "Dynamic Source Routing Protocol"},
	49:                 {Sym: "bna", Description: "BNA"},
	50:                 {Sym: "esp", Description: "encapsulating security payload"},
	51:                 {Sym: "ah", Description: "authentication header"},
	52:                 {Sym: "i-nlsp", Description: "Integrated Net Layer Security TUBA"},
	53:                 {Sym: "swipe", Description: "IP with Encryption"},
	54:                 {Sym: "narp", Description: "NBMA Address Resolution Protocol"},
	55:                 {Sym: "mobile", Description: "IP Mobility"},
	56:                 {Sym: "tlsp", Description: "Transport Layer Security Protocol"},
	57:                 {Sym: "skip", Description: "SKIP"},
	IPv4ProtocolICMPv6: {Sym: "ipv6-icmp", Description: "ICMP for IPv6"},
	59:                 {Sym: "ipv6-nonxt", Description: "no next header for ipv6"},
	60:                 {Sym: "ipv6-opts", Description: "destination options for ipv6"},
	62:                 {Sym: "cftp", Description: "CFTP"},
	64:                 {Sym: "sat-expak", Description: "SATNET and Backroom EXPAK"},
	65:                 {Sym: "kryptolan", Description: "Kryptolan"},
	66:                 {Sym: "rvd", Description: "MIT Remote Virtual Disk Protocol"},
	67:                 {Sym: "ippc", Description: "Internet Pluribus Packet Core"},
	69:                 {Sym: "sat-mon", Description: "SATNET Monitoring"},
	70:                 {Sym: "visa", Description: "VISA Protocol"},
	71:                 {Sym: "ipcv", Description: "Internet Packet Core Utility"},
	72:                 {Sym: "cpnx", Description: "Computer Protocol Network Executive"},
	73:                 {Sym: "cphb", Description: "Computer Protocol Heart Beat"},
	74:                 {Sym: "wsn", Description: "Wang Span Network"},
	75:                 {Sym: "pvp", Description: "Packet Video Protocol"},
	76:                 {Sym: "br-sat-mon", Description: "Backroom SATNET Monitoring"},
	77:                 {Sym: "sun-nd", Description: "SUN ND PROTOCOL-Temporary"},
	78:                 {Sym: "wb-mon", Description: "WIDEBAND Monitoring"},
	79:                 {Sym: "wb-expak", Description: "WIDEBAND EXPAK"},
	80:                 {Sym: "iso-ip", Description: "ISO Internet Protocol"},
	81:                 {Sym: "vmtp", Description: "Versatile Message Transport"},
	82:                 {Sym: "secure-vmtp", Description: "SECURE-VMTP"},
	83:                 {Sym: "vines", Description: "VINES"},
	84:                 {Sym: "ttp", Description: "TTP"},
	85:                 {Sym: "nsfnet-igp", Description: "NSFNET-IGP"},
	86:                 {Sym: "dgp", Description: "Dissimilar Gateway Protocol"},
	87:                 {Sym: "tcf", Description: "TCF"},
	88:                 {Sym: "eigrp", Description: "Enhanced Interior Routing Protocol (Cisco)"},
	89:                 {Sym: "ospf", Description: "Open Shortest Path First IGP"},
	90:                 {Sym: "sprite-rpc", Description: "Sprite RPC Protocol"},
	91:                 {Sym: "larp", Description: "Locus Address Resolution Protocol"},
	92:                 {Sym: "mtp", Description: "Multicast Transport Protocol"},
	93:                 {Sym: "25", Description: "AX.25 Frames"},
	94:                 {Sym: "ipip", Description: "Yet Another IP encapsulation"},
	95:                 {Sym: "micp", Description: "Mobile Internetworking Control Pro"},
	96:                 {Sym: "scc-sp", Description: "Semaphore Communications Sec. Pro"},
	97:                 {Sym: "etherip", Description: "Ethernet-within-IP Encapsulation"},
	98:                 {Sym: "encap", Description: "Yet Another IP encapsulation"},
	100:                {Sym: "gmtp", Description: "GMTP"},
	101:                {Sym: "ifmp", Description: "Ipsilon Flow Management Protocol"},
	102:                {Sym: "pnni", Description: "PNNI over IP"},
	103:                {Sym: "pim", Description: "Protocol Independent Multicast"},
	104:                {Sym: "aris", Description: "ARIS"},
	105:                {Sym: "scps", Description: "SCPS"},
	106:                {Sym: "qnx", Description: "QNX"},
	107:                {Sym: "n", Description: "Active Networks"},
	108:                {Sym: "ipcomp", Description: "IP Payload Compression Protocol"},
	109:                {Sym: "snp", Description: "Sitara Networks Protocol"},
	110:                {Sym: "compaq-peer", Description: "Compaq Peer Protocol"},
	111:                {Sym: "ipx-in-ip", Description: "IPX in IP"},
	112:                {Sym: "carp", Description: "Common Address Redundancy Protocol"},
	113:                {Sym: "pgm", Description: "PGM Reliable Transport Protocol"},
	115:                {Sym: "l2tp", Description: "Layer Two Tunneling Protocol"},
	116:                {Sym: "ddx", Description: "D-II Data Exchange"},
	117:                {Sym: "iatp", Description: "Interactive Agent Transfer Protocol"},
	118:                {Sym: "stp", Description: "Schedule Transfer Protocol"},
	119:                {Sym: "srp", Description: "SpectraLink Radio Protocol"},
	120:                {Sym: "uti", Description: "UTI"},
	121:                {Sym: "smp", Description: "Simple Message Protocol"},
	122:                {Sym: "sm", Description: "SM"},
	123:                {Sym: "ptp", Description: "Performance Transparency Protocol"},
	124:                {Sym: "isis", Description: "ISIS over IPv4"},
	126:                {Sym: "crtp", Description: "Combat Radio Transport Protocol"},
	127:                {Sym: "crudp", Description: "Combat Radio User Datagram"},
	130:                {Sym: "sps", Description: "Secure Packet Shield"},
	131:                {Sym: "pipe", Description: "Private IP Encapsulation within IP"},
	132:                {Sym: "sctp", Description: "Stream Control Transmission Protocol"},
	133:                {Sym: "fc", Description: "Fibre Channel"},
	134:                {Sym: "rsvp-e2e-ignore", Description: "Aggregation of RSVP for IP reservations"},
	135:                {Sym: "mobility-header", Description: "Mobility Support in IPv6"},
	136:                {Sym: "udplite", Description: "The UDP-Lite Protocol"},
	137:                {Sym: "mpls-in-ip", Description: "Encapsulating MPLS in IP"},
	138:                {Sym: "manet", Description: "MANET Protocols (RFC5498)"},
	139:                {Sym: "hip", Description: "Host Identity Protocol (RFC5201)"},
	140:                {Sym: "shim6", Description: "Shim6 Protocol (RFC5533)"},
	141:                {Sym: "wesp", Description: "Wrapped Encapsulating Security Payload (RFC5840)"},
	142:                {Sym: "rohc", Description: "Robust Header Compression (RFC5858)"},
	240:                {Sym: "pfsync", Description: "PF Synchronization"},
	258:                {Sym: "divert", Description: "Divert pseudo-protocol [non IANA]"},
}

// based on etc/services from Darwin/FreeBSD
// cat /etc/services | grep -v '^#'  | jq -rR 'capture("(?<name>[\\w\\d-]+)\\s+(?<port>\\d+)/(?<proto>\\w+)\\s+.*#\\s+(?<desc>.*)") | select(.proto=="udp") | "\(.port): {Sym: \(.name|tojson), Description: \(.desc|tojson)},"'
// current truncated to < 1024

const (
	UDPPortDomain = 53
	UDPPortMDNS   = 5353
)

var UDPPortMap = scalar.UintMap{
	1:             {Sym: "tcpmux", Description: "TCP Port Service Multiplexer"},
	2:             {Sym: "compressnet", Description: "Management Utility"},
	3:             {Sym: "compressnet", Description: "Compression Process"},
	5:             {Sym: "rje", Description: "Remote Job Entry"},
	7:             {Sym: "echo", Description: "Echo"},
	9:             {Sym: "discard", Description: "Discard"},
	11:            {Sym: "systat", Description: "Active Users"},
	13:            {Sym: "daytime", Description: "Daytime (RFC 867)"},
	17:            {Sym: "qotd", Description: "Quote of the Day"},
	18:            {Sym: "msp", Description: "Message Send Protocol"},
	19:            {Sym: "chargen", Description: "Character Generator"},
	20:            {Sym: "ftp-data", Description: "File Transfer [Default Data]"},
	21:            {Sym: "ftp", Description: "File Transfer [Control]"},
	22:            {Sym: "ssh", Description: "SSH Remote Login Protocol"},
	23:            {Sym: "telnet", Description: "Telnet"},
	25:            {Sym: "smtp", Description: "Simple Mail Transfer"},
	27:            {Sym: "nsw-fe", Description: "NSW User System FE"},
	29:            {Sym: "msg-icp", Description: "MSG ICP"},
	31:            {Sym: "msg-auth", Description: "MSG Authentication"},
	33:            {Sym: "dsp", Description: "Display Support Protocol"},
	37:            {Sym: "time", Description: "Time"},
	38:            {Sym: "rap", Description: "Route Access Protocol"},
	39:            {Sym: "rlp", Description: "Resource Location Protocol"},
	41:            {Sym: "graphics", Description: "Graphics"},
	42:            {Sym: "name", Description: "Host Name Server"},
	44:            {Sym: "mpm-flags", Description: "MPM FLAGS Protocol"},
	45:            {Sym: "mpm", Description: "Message Processing Module [recv]"},
	46:            {Sym: "mpm-snd", Description: "MPM [default send]"},
	47:            {Sym: "ni-ftp", Description: "NI FTP"},
	48:            {Sym: "auditd", Description: "Digital Audit Daemon"},
	49:            {Sym: "tacacs", Description: "Login Host Protocol (TACACS)"},
	50:            {Sym: "re-mail-ck", Description: "Remote Mail Checking Protocol"},
	51:            {Sym: "la-maint", Description: "IMP Logical Address Maintenance"},
	52:            {Sym: "xns-time", Description: "XNS Time Protocol"},
	UDPPortDomain: {Sym: "domain", Description: "Domain Name Server"},
	54:            {Sym: "xns-ch", Description: "XNS Clearinghouse"},
	55:            {Sym: "isi-gl", Description: "ISI Graphics Language"},
	56:            {Sym: "xns-auth", Description: "XNS Authentication"},
	58:            {Sym: "xns-mail", Description: "XNS Mail"},
	61:            {Sym: "ni-mail", Description: "NI MAIL"},
	62:            {Sym: "acas", Description: "ACA Services"},
	64:            {Sym: "covia", Description: "Communications Integrator (CI)"},
	65:            {Sym: "tacacs-ds", Description: "TACACS-Database Service"},
	66:            {Sym: "net", Description: "Oracle SQL*NET"},
	67:            {Sym: "bootps", Description: "Bootstrap Protocol Server"},
	68:            {Sym: "bootpc", Description: "Bootstrap Protocol Client"},
	69:            {Sym: "tftp", Description: "Trivial File Transfer"},
	70:            {Sym: "gopher", Description: "Gopher"},
	71:            {Sym: "netrjs-1", Description: "Remote Job Service"},
	72:            {Sym: "netrjs-2", Description: "Remote Job Service"},
	73:            {Sym: "netrjs-3", Description: "Remote Job Service"},
	74:            {Sym: "netrjs-4", Description: "Remote Job Service"},
	76:            {Sym: "deos", Description: "Distributed External Object Store"},
	78:            {Sym: "vettcp", Description: "vettcp"},
	79:            {Sym: "finger", Description: "Finger"},
	80:            {Sym: "http", Description: "World Wide Web HTTP"},
	81:            {Sym: "hosts2-ns", Description: "HOSTS2 Name Server"},
	82:            {Sym: "xfer", Description: "XFER Utility"},
	83:            {Sym: "mit-ml-dev", Description: "MIT ML Device"},
	84:            {Sym: "ctf", Description: "Common Trace Facility"},
	85:            {Sym: "mit-ml-dev", Description: "MIT ML Device"},
	86:            {Sym: "mfcobol", Description: "Micro Focus Cobol"},
	88:            {Sym: "kerberos", Description: "Kerberos"},
	89:            {Sym: "su-mit-tg", Description: "SU/MIT Telnet Gateway"},
	90:            {Sym: "dnsix", Description: "DNSIX Securit Attribute Token Map"},
	91:            {Sym: "mit-dov", Description: "MIT Dover Spooler"},
	92:            {Sym: "npp", Description: "Network Printing Protocol"},
	93:            {Sym: "dcp", Description: "Device Control Protocol"},
	94:            {Sym: "objcall", Description: "Tivoli Object Dispatcher"},
	95:            {Sym: "supdup", Description: "SUPDUP"},
	96:            {Sym: "dixie", Description: "DIXIE Protocol Specification"},
	97:            {Sym: "swift-rvf", Description: "Swift Remote Virtural File Protocol"},
	98:            {Sym: "tacnews", Description: "TAC News"},
	99:            {Sym: "metagram", Description: "Metagram Relay"},
	101:           {Sym: "hostname", Description: "NIC Host Name Server"},
	102:           {Sym: "iso-tsap", Description: "ISO-TSAP Class 0"},
	103:           {Sym: "gppitnp", Description: "Genesis Point-to-Point Trans Net"},
	104:           {Sym: "acr-nema", Description: "ACR-NEMA Digital Imag. & Comm. 300"},
	105:           {Sym: "cso", Description: "CCSO name server protocol"},
	106:           {Sym: "3com-tsmux", Description: "3COM-TSMUX"},
	107:           {Sym: "rtelnet", Description: "Remote Telnet Service"},
	108:           {Sym: "snagas", Description: "SNA Gateway Access Server"},
	109:           {Sym: "pop2", Description: "Post Office Protocol - Version 2"},
	110:           {Sym: "pop3", Description: "Post Office Protocol - Version 3"},
	111:           {Sym: "sunrpc", Description: "SUN Remote Procedure Call"},
	112:           {Sym: "mcidas", Description: "McIDAS Data Transmission Protocol"},
	113:           {Sym: "auth", Description: "Authentication Service"},
	114:           {Sym: "audionews", Description: "Audio News Multicast"},
	115:           {Sym: "sftp", Description: "Simple File Transfer Protocol"},
	116:           {Sym: "ansanotify", Description: "ANSA REX Notify"},
	117:           {Sym: "uucp-path", Description: "UUCP Path Service"},
	118:           {Sym: "sqlserv", Description: "SQL Services"},
	119:           {Sym: "nntp", Description: "Network News Transfer Protocol"},
	120:           {Sym: "cfdptkt", Description: "CFDPTKT"},
	121:           {Sym: "erpc", Description: "Encore Expedited Remote Pro.Call"},
	122:           {Sym: "smakynet", Description: "SMAKYNET"},
	123:           {Sym: "ntp", Description: "Network Time Protocol"},
	124:           {Sym: "ansatrader", Description: "ANSA REX Trader"},
	125:           {Sym: "locus-map", Description: "Locus PC-Interface Net Map Ser"},
	126:           {Sym: "nxedit", Description: "NXEdit"},
	127:           {Sym: "locus-con", Description: "Locus PC-Interface Conn Server"},
	128:           {Sym: "gss-xlicen", Description: "GSS X License Verification"},
	129:           {Sym: "pwdgen", Description: "Password Generator Protocol"},
	130:           {Sym: "cisco-fna", Description: "cisco FNATIVE"},
	131:           {Sym: "cisco-tna", Description: "cisco TNATIVE"},
	132:           {Sym: "cisco-sys", Description: "cisco SYSMAINT"},
	133:           {Sym: "statsrv", Description: "Statistics Service"},
	134:           {Sym: "ingres-net", Description: "INGRES-NET Service"},
	135:           {Sym: "epmap", Description: "DCE endpoint resolution"},
	136:           {Sym: "profile", Description: "PROFILE Naming System"},
	137:           {Sym: "netbios-ns", Description: "NETBIOS Name Service"},
	138:           {Sym: "netbios-dgm", Description: "NETBIOS Datagram Service"},
	139:           {Sym: "netbios-ssn", Description: "NETBIOS Session Service"},
	140:           {Sym: "emfis-data", Description: "EMFIS Data Service"},
	141:           {Sym: "emfis-cntl", Description: "EMFIS Control Service"},
	142:           {Sym: "bl-idm", Description: "Britton-Lee IDM"},
	143:           {Sym: "imap", Description: "Internet Message Access Protocol"},
	144:           {Sym: "uma", Description: "Universal Management Architecture"},
	145:           {Sym: "uaac", Description: "UAAC Protocol"},
	146:           {Sym: "iso-tp0", Description: "ISO-IP0"},
	147:           {Sym: "iso-ip", Description: "ISO-IP"},
	148:           {Sym: "jargon", Description: "Jargon"},
	149:           {Sym: "aed-512", Description: "AED 512 Emulation Service"},
	150:           {Sym: "sql-net", Description: "SQL-NET"},
	151:           {Sym: "hems", Description: "HEMS"},
	152:           {Sym: "bftp", Description: "Background File Transfer Program"},
	153:           {Sym: "sgmp", Description: "SGMP"},
	154:           {Sym: "netsc-prod", Description: "NETSC"},
	155:           {Sym: "netsc-dev", Description: "NETSC"},
	156:           {Sym: "sqlsrv", Description: "SQL Service"},
	157:           {Sym: "knet-cmp", Description: "KNET/VM Command/Message Protocol"},
	158:           {Sym: "pcmail-srv", Description: "PCMail Server"},
	159:           {Sym: "nss-routing", Description: "NSS-Routing"},
	160:           {Sym: "sgmp-traps", Description: "SGMP-TRAPS"},
	161:           {Sym: "snmp", Description: "SNMP"},
	162:           {Sym: "snmptrap", Description: "SNMPTRAP"},
	163:           {Sym: "cmip-man", Description: "CMIP/TCP Manager"},
	164:           {Sym: "cmip-agent", Description: "CMIP/TCP Agent"},
	165:           {Sym: "xns-courier", Description: "Xerox"},
	166:           {Sym: "s-net", Description: "Sirius Systems"},
	167:           {Sym: "namp", Description: "NAMP"},
	168:           {Sym: "rsvd", Description: "RSVD"},
	169:           {Sym: "send", Description: "SEND"},
	170:           {Sym: "print-srv", Description: "Network PostScript"},
	171:           {Sym: "multiplex", Description: "Network Innovations Multiplex"},
	172:           {Sym: "1", Description: "Network Innovations CL/1"},
	173:           {Sym: "xyplex-mux", Description: "Xyplex"},
	174:           {Sym: "mailq", Description: "MAILQ"},
	175:           {Sym: "vmnet", Description: "VMNET"},
	176:           {Sym: "genrad-mux", Description: "GENRAD-MUX"},
	177:           {Sym: "xdmcp", Description: "X Display Manager Control Protocol"},
	178:           {Sym: "nextstep", Description: "NextStep Window Server"},
	179:           {Sym: "bgp", Description: "Border Gateway Protocol"},
	180:           {Sym: "ris", Description: "Intergraph"},
	181:           {Sym: "unify", Description: "Unify"},
	182:           {Sym: "audit", Description: "Unisys Audit SITP"},
	183:           {Sym: "ocbinder", Description: "OCBinder"},
	184:           {Sym: "ocserver", Description: "OCServer"},
	185:           {Sym: "remote-kis", Description: "Remote-KIS"},
	186:           {Sym: "kis", Description: "KIS Protocol"},
	187:           {Sym: "aci", Description: "Application Communication Interface"},
	188:           {Sym: "mumps", Description: "Plus Five's MUMPS"},
	189:           {Sym: "qft", Description: "Queued File Transport"},
	190:           {Sym: "gacp", Description: "Gateway Access Control Protocol"},
	191:           {Sym: "prospero", Description: "Prospero Directory Service"},
	192:           {Sym: "osu-nms", Description: "OSU Network Monitoring System"},
	193:           {Sym: "srmp", Description: "Spider Remote Monitoring Protocol"},
	194:           {Sym: "irc", Description: "Internet Relay Chat Protocol"},
	195:           {Sym: "dn6-nlm-aud", Description: "DNSIX Network Level Module Audit"},
	196:           {Sym: "dn6-smm-red", Description: "DNSIX Session Mgt Module Audit Redir"},
	197:           {Sym: "dls", Description: "Directory Location Service"},
	198:           {Sym: "dls-mon", Description: "Directory Location Service Monitor"},
	199:           {Sym: "smux", Description: "SMUX"},
	200:           {Sym: "src", Description: "IBM System Resource Controller"},
	201:           {Sym: "at-rtmp", Description: "AppleTalk Routing Maintenance"},
	202:           {Sym: "at-nbp", Description: "AppleTalk Name Binding"},
	203:           {Sym: "at-3", Description: "AppleTalk Unused"},
	204:           {Sym: "at-echo", Description: "AppleTalk Echo"},
	205:           {Sym: "at-5", Description: "AppleTalk Unused"},
	206:           {Sym: "at-zis", Description: "AppleTalk Zone Information"},
	207:           {Sym: "at-7", Description: "AppleTalk Unused"},
	208:           {Sym: "at-8", Description: "AppleTalk Unused"},
	209:           {Sym: "qmtp", Description: "The Quick Mail Transfer Protocol"},
	210:           {Sym: "50", Description: "ANSI Z39.50"},
	211:           {Sym: "g", Description: "Texas Instruments 914C/G Terminal"},
	212:           {Sym: "anet", Description: "ATEXSSTR"},
	213:           {Sym: "ipx", Description: "IPX"},
	214:           {Sym: "vmpwscs", Description: "VM PWSCS"},
	215:           {Sym: "softpc", Description: "Insignia Solutions"},
	216:           {Sym: "cailic", Description: "Computer Associates Int'l License Server"},
	217:           {Sym: "dbase", Description: "dBASE Unix"},
	218:           {Sym: "mpp", Description: "Netix Message Posting Protocol"},
	219:           {Sym: "uarps", Description: "Unisys ARPs"},
	220:           {Sym: "imap3", Description: "Interactive Mail Access Protocol v3"},
	221:           {Sym: "fln-spx", Description: "Berkeley rlogind with SPX auth"},
	222:           {Sym: "rsh-spx", Description: "Berkeley rshd with SPX auth"},
	223:           {Sym: "cdc", Description: "Certificate Distribution Center"},
	224:           {Sym: "masqdialer", Description: "masqdialer"},
	242:           {Sym: "direct", Description: "Direct"},
	243:           {Sym: "sur-meas", Description: "Survey Measurement"},
	244:           {Sym: "inbusiness", Description: "inbusiness"},
	245:           {Sym: "link", Description: "LINK"},
	246:           {Sym: "dsp3270", Description: "Display Systems Protocol"},
	247:           {Sym: "subntbcst_tftp", Description: "SUBNTBCST_TFTP"},
	248:           {Sym: "bhfhs", Description: "bhfhs"},
	256:           {Sym: "rap", Description: "RAP"},
	257:           {Sym: "set", Description: "Secure Electronic Transaction"},
	258:           {Sym: "yak-chat", Description: "Yak Winsock Personal Chat"},
	259:           {Sym: "esro-gen", Description: "Efficient Short Remote Operations"},
	260:           {Sym: "openport", Description: "Openport"},
	261:           {Sym: "nsiiops", Description: "IIOP Name Service over TLS/SSL"},
	262:           {Sym: "arcisdms", Description: "Arcisdms"},
	263:           {Sym: "hdap", Description: "HDAP"},
	264:           {Sym: "bgmp", Description: "BGMP"},
	265:           {Sym: "x-bone-ctl", Description: "X-Bone CTL"},
	266:           {Sym: "sst", Description: "SCSI on ST"},
	267:           {Sym: "td-service", Description: "Tobit David Service Layer"},
	268:           {Sym: "td-replica", Description: "Tobit David Replica"},
	280:           {Sym: "http-mgmt", Description: "http-mgmt"},
	281:           {Sym: "personal-link", Description: "Personal Link"},
	282:           {Sym: "cableport-ax", Description: "Cable Port A/X"},
	283:           {Sym: "rescap", Description: "rescap"},
	284:           {Sym: "corerjd", Description: "corerjd"},
	286:           {Sym: "fxp-1", Description: "FXP-1"},
	287:           {Sym: "k-block", Description: "K-BLOCK"},
	308:           {Sym: "novastorbakcup", Description: "Novastor Backup"},
	309:           {Sym: "entrusttime", Description: "EntrustTime"},
	310:           {Sym: "bhmds", Description: "bhmds"},
	311:           {Sym: "asip-webadmin", Description: "AppleShare IP WebAdmin"},
	312:           {Sym: "vslmp", Description: "VSLMP"},
	313:           {Sym: "magenta-logic", Description: "Magenta Logic"},
	314:           {Sym: "opalis-robot", Description: "Opalis Robot"},
	315:           {Sym: "dpsi", Description: "DPSI"},
	316:           {Sym: "decauth", Description: "decAuth"},
	317:           {Sym: "zannet", Description: "Zannet"},
	318:           {Sym: "pkix-timestamp", Description: "PKIX TimeStamp"},
	319:           {Sym: "ptp-event", Description: "PTP Event"},
	320:           {Sym: "ptp-general", Description: "PTP General"},
	321:           {Sym: "pip", Description: "PIP"},
	322:           {Sym: "rtsps", Description: "RTSPS"},
	333:           {Sym: "texar", Description: "Texar Security Port"},
	344:           {Sym: "pdap", Description: "Prospero Data Access Protocol"},
	345:           {Sym: "pawserv", Description: "Perf Analysis Workbench"},
	346:           {Sym: "zserv", Description: "Zebra server"},
	347:           {Sym: "fatserv", Description: "Fatmen Server"},
	348:           {Sym: "csi-sgwp", Description: "Cabletron Management Protocol"},
	349:           {Sym: "mftp", Description: "mftp"},
	350:           {Sym: "matip-type-a", Description: "MATIP Type A"},
	351:           {Sym: "matip-type-b", Description: "MATIP Type B"},
	352:           {Sym: "dtag-ste-sb", Description: "DTAG"},
	353:           {Sym: "ndsauth", Description: "NDSAUTH"},
	354:           {Sym: "bh611", Description: "bh611"},
	355:           {Sym: "datex-asn", Description: "DATEX-ASN"},
	356:           {Sym: "cloanto-net-1", Description: "Cloanto Net 1"},
	357:           {Sym: "bhevent", Description: "bhevent"},
	358:           {Sym: "shrinkwrap", Description: "Shrinkwrap"},
	359:           {Sym: "nsrmp", Description: "Network Security Risk Management Protocol"},
	360:           {Sym: "scoi2odialog", Description: "scoi2odialog"},
	361:           {Sym: "semantix", Description: "Semantix"},
	362:           {Sym: "srssend", Description: "SRS Send"},
	363:           {Sym: "rsvp_tunnel", Description: "RSVP Tunnel"},
	364:           {Sym: "aurora-cmgr", Description: "Aurora CMGR"},
	365:           {Sym: "dtk", Description: "DTK"},
	366:           {Sym: "odmr", Description: "ODMR"},
	367:           {Sym: "mortgageware", Description: "MortgageWare"},
	368:           {Sym: "qbikgdp", Description: "QbikGDP"},
	369:           {Sym: "rpc2portmap", Description: "rpc2portmap"},
	370:           {Sym: "codaauth2", Description: "codaauth2"},
	371:           {Sym: "clearcase", Description: "Clearcase"},
	372:           {Sym: "ulistproc", Description: "ListProcessor"},
	373:           {Sym: "legent-1", Description: "Legent Corporation"},
	374:           {Sym: "legent-2", Description: "Legent Corporation"},
	375:           {Sym: "hassle", Description: "Hassle"},
	376:           {Sym: "nip", Description: "Amiga Envoy Network Inquiry Proto"},
	377:           {Sym: "tnETOS", Description: "NEC Corporation"},
	378:           {Sym: "dsETOS", Description: "NEC Corporation"},
	379:           {Sym: "is99c", Description: "TIA/EIA/IS-99 modem client"},
	380:           {Sym: "is99s", Description: "TIA/EIA/IS-99 modem server"},
	381:           {Sym: "hp-collector", Description: "hp performance data collector"},
	382:           {Sym: "hp-managed-node", Description: "hp performance data managed node"},
	383:           {Sym: "hp-alarm-mgr", Description: "hp performance data alarm manager"},
	384:           {Sym: "arns", Description: "A Remote Network Server System"},
	385:           {Sym: "ibm-app", Description: "IBM Application"},
	386:           {Sym: "asa", Description: "ASA Message Router Object Def"},
	387:           {Sym: "aurp", Description: "Appletalk Update-Based Routing Pro"},
	388:           {Sym: "unidata-ldm", Description: "Unidata LDM"},
	389:           {Sym: "ldap", Description: "Lightweight Directory Access Protocol"},
	390:           {Sym: "uis", Description: "UIS"},
	391:           {Sym: "synotics-relay", Description: "SynOptics SNMP Relay Port"},
	392:           {Sym: "synotics-broker", Description: "SynOptics Port Broker Port"},
	393:           {Sym: "meta5", Description: "Meta5"},
	394:           {Sym: "embl-ndt", Description: "EMBL Nucleic Data Transfer"},
	395:           {Sym: "netcp", Description: "NETscout Control Protocol"},
	396:           {Sym: "netware-ip", Description: "Novell Netware over IP"},
	397:           {Sym: "mptn", Description: "Multi Protocol Trans. Net"},
	398:           {Sym: "kryptolan", Description: "Kryptolan"},
	399:           {Sym: "iso-tsap-c2", Description: "ISO Transport Class 2 Non-Control over UDP"},
	400:           {Sym: "work-sol", Description: "Workstation Solutions"},
	401:           {Sym: "ups", Description: "Uninterruptible Power Supply"},
	402:           {Sym: "genie", Description: "Genie Protocol"},
	403:           {Sym: "decap", Description: "decap"},
	404:           {Sym: "nced", Description: "nced"},
	405:           {Sym: "ncld", Description: "ncld"},
	406:           {Sym: "imsp", Description: "Interactive Mail Support Protocol"},
	407:           {Sym: "timbuktu", Description: "Timbuktu"},
	408:           {Sym: "prm-sm", Description: "Prospero Resource Manager Sys. Man"},
	409:           {Sym: "prm-nm", Description: "Prospero Resource Manager Node Man"},
	410:           {Sym: "decladebug", Description: "DECLadebug Remote Debug Protocol"},
	411:           {Sym: "rmt", Description: "Remote MT Protocol"},
	412:           {Sym: "synoptics-trap", Description: "Trap Convention Port"},
	413:           {Sym: "smsp", Description: "Storage Management Services Protocol"},
	414:           {Sym: "infoseek", Description: "InfoSeek"},
	415:           {Sym: "bnet", Description: "BNet"},
	416:           {Sym: "silverplatter", Description: "Silverplatter"},
	417:           {Sym: "onmux", Description: "Onmux"},
	418:           {Sym: "hyper-g", Description: "Hyper-G"},
	419:           {Sym: "ariel1", Description: "Ariel 1"},
	420:           {Sym: "smpte", Description: "SMPTE"},
	421:           {Sym: "ariel2", Description: "Ariel 2"},
	422:           {Sym: "ariel3", Description: "Ariel 3"},
	423:           {Sym: "opc-job-start", Description: "IBM Operations Planning and Control Start"},
	424:           {Sym: "opc-job-track", Description: "IBM Operations Planning and Control Track"},
	425:           {Sym: "icad-el", Description: "ICAD"},
	426:           {Sym: "smartsdp", Description: "smartsdp"},
	427:           {Sym: "svrloc", Description: "Server Location"},
	428:           {Sym: "ocs_cmu", Description: "OCS_CMU"},
	429:           {Sym: "ocs_amu", Description: "OCS_AMU"},
	430:           {Sym: "utmpsd", Description: "UTMPSD"},
	431:           {Sym: "utmpcd", Description: "UTMPCD"},
	432:           {Sym: "iasd", Description: "IASD"},
	433:           {Sym: "nnsp", Description: "NNSP"},
	434:           {Sym: "mobileip-agent", Description: "MobileIP-Agent"},
	435:           {Sym: "mobilip-mn", Description: "MobilIP-MN"},
	436:           {Sym: "dna-cml", Description: "DNA-CML"},
	437:           {Sym: "comscm", Description: "comscm"},
	438:           {Sym: "dsfgw", Description: "dsfgw"},
	439:           {Sym: "dasp", Description: "dasp"},
	440:           {Sym: "sgcp", Description: "sgcp"},
	441:           {Sym: "decvms-sysmgt", Description: "decvms-sysmgt"},
	442:           {Sym: "cvc_hostd", Description: "cvc_hostd"},
	443:           {Sym: "https", Description: "http protocol over TLS/SSL"},
	444:           {Sym: "snpp", Description: "Simple Network Paging Protocol"},
	445:           {Sym: "microsoft-ds", Description: "Microsoft-DS"},
	446:           {Sym: "ddm-rdb", Description: "DDM-RDB"},
	447:           {Sym: "ddm-dfm", Description: "DDM-RFM"},
	448:           {Sym: "ddm-ssl", Description: "DDM-SSL"},
	449:           {Sym: "as-servermap", Description: "AS Server Mapper"},
	450:           {Sym: "tserver", Description: "Computer Supported Telecomunication Applications"},
	451:           {Sym: "sfs-smp-net", Description: "Cray Network Semaphore server"},
	452:           {Sym: "sfs-config", Description: "Cray SFS config server"},
	453:           {Sym: "creativeserver", Description: "CreativeServer"},
	454:           {Sym: "contentserver", Description: "ContentServer"},
	455:           {Sym: "creativepartnr", Description: "CreativePartnr"},
	456:           {Sym: "macon-udp", Description: "macon-udp"},
	457:           {Sym: "scohelp", Description: "scohelp"},
	458:           {Sym: "appleqtc", Description: "apple quick time"},
	459:           {Sym: "ampr-rcmd", Description: "ampr-rcmd"},
	460:           {Sym: "skronk", Description: "skronk"},
	461:           {Sym: "datasurfsrv", Description: "DataRampSrv"},
	462:           {Sym: "datasurfsrvsec", Description: "DataRampSrvSec"},
	463:           {Sym: "alpes", Description: "alpes"},
	464:           {Sym: "kpasswd", Description: "kpasswd"},
	465:           {Sym: "igmpv3lite", Description: "IGMP over UDP for SSM"},
	466:           {Sym: "digital-vrc", Description: "digital-vrc"},
	467:           {Sym: "mylex-mapd", Description: "mylex-mapd"},
	468:           {Sym: "photuris", Description: "proturis"},
	469:           {Sym: "rcp", Description: "Radio Control Protocol"},
	470:           {Sym: "scx-proxy", Description: "scx-proxy"},
	471:           {Sym: "mondex", Description: "Mondex"},
	472:           {Sym: "ljk-login", Description: "ljk-login"},
	473:           {Sym: "hybrid-pop", Description: "hybrid-pop"},
	474:           {Sym: "tn-tl-w2", Description: "tn-tl-w2"},
	475:           {Sym: "tcpnethaspsrv", Description: "tcpnethaspsrv"},
	476:           {Sym: "tn-tl-fd1", Description: "tn-tl-fd1"},
	477:           {Sym: "ss7ns", Description: "ss7ns"},
	478:           {Sym: "spsc", Description: "spsc"},
	479:           {Sym: "iafserver", Description: "iafserver"},
	480:           {Sym: "iafdbase", Description: "iafdbase"},
	481:           {Sym: "ph", Description: "Ph service"},
	482:           {Sym: "bgs-nsi", Description: "bgs-nsi"},
	483:           {Sym: "ulpnet", Description: "ulpnet"},
	484:           {Sym: "integra-sme", Description: "Integra Software Management Environment"},
	485:           {Sym: "powerburst", Description: "Air Soft Power Burst"},
	486:           {Sym: "avian", Description: "avian"},
	487:           {Sym: "saft", Description: "saft Simple Asynchronous File Transfer"},
	488:           {Sym: "gss-http", Description: "gss-http"},
	489:           {Sym: "nest-protocol", Description: "nest-protocol"},
	490:           {Sym: "micom-pfs", Description: "micom-pfs"},
	491:           {Sym: "go-login", Description: "go-login"},
	492:           {Sym: "ticf-1", Description: "Transport Independent Convergence for FNA"},
	493:           {Sym: "ticf-2", Description: "Transport Independent Convergence for FNA"},
	494:           {Sym: "pov-ray", Description: "POV-Ray"},
	495:           {Sym: "intecourier", Description: "intecourier"},
	496:           {Sym: "pim-rp-disc", Description: "PIM-RP-DISC"},
	497:           {Sym: "dantz", Description: "dantz"},
	498:           {Sym: "siam", Description: "siam"},
	499:           {Sym: "iso-ill", Description: "ISO ILL Protocol"},
	500:           {Sym: "isakmp", Description: "isakmp"},
	501:           {Sym: "stmf", Description: "STMF"},
	502:           {Sym: "asa-appl-proto", Description: "asa-appl-proto"},
	503:           {Sym: "intrinsa", Description: "Intrinsa"},
	504:           {Sym: "citadel", Description: "citadel"},
	505:           {Sym: "mailbox-lm", Description: "mailbox-lm"},
	506:           {Sym: "ohimsrv", Description: "ohimsrv"},
	507:           {Sym: "crs", Description: "crs"},
	508:           {Sym: "xvttp", Description: "xvttp"},
	509:           {Sym: "snare", Description: "snare"},
	510:           {Sym: "fcp", Description: "FirstClass Protocol"},
	511:           {Sym: "passgo", Description: "PassGo"},
	512:           {Sym: "comsat"},
	513:           {Sym: "who", Description: "maintains data bases showing who's"},
	514:           {Sym: "syslog"},
	515:           {Sym: "printer", Description: "spooler"},
	516:           {Sym: "videotex", Description: "videotex"},
	517:           {Sym: "talk", Description: "like tenex link, but across"},
	518:           {Sym: "ntalk"},
	519:           {Sym: "utime", Description: "unixtime"},
	520:           {Sym: "router", Description: "local routing process (on site);"},
	521:           {Sym: "ripng", Description: "ripng"},
	522:           {Sym: "ulp", Description: "ULP"},
	523:           {Sym: "ibm-db2", Description: "IBM-DB2"},
	524:           {Sym: "ncp", Description: "NCP"},
	525:           {Sym: "timed", Description: "timeserver"},
	526:           {Sym: "tempo", Description: "newdate"},
	527:           {Sym: "stx", Description: "Stock IXChange"},
	528:           {Sym: "custix", Description: "Customer IXChange"},
	529:           {Sym: "irc-serv", Description: "IRC-SERV"},
	530:           {Sym: "courier", Description: "rpc"},
	531:           {Sym: "conference", Description: "chat"},
	532:           {Sym: "netnews", Description: "readnews"},
	533:           {Sym: "netwall", Description: "for emergency broadcasts"},
	534:           {Sym: "mm-admin", Description: "MegaMedia Admin"},
	535:           {Sym: "iiop", Description: "iiop"},
	536:           {Sym: "opalis-rdv", Description: "opalis-rdv"},
	537:           {Sym: "nmsp", Description: "Networked Media Streaming Protocol"},
	538:           {Sym: "gdomap", Description: "gdomap"},
	539:           {Sym: "apertus-ldp", Description: "Apertus Technologies Load Determination"},
	540:           {Sym: "uucp", Description: "uucpd\t\t"},
	541:           {Sym: "uucp-rlogin", Description: "uucp-rlogin"},
	542:           {Sym: "commerce", Description: "commerce"},
	543:           {Sym: "klogin"},
	544:           {Sym: "kshell", Description: "krcmd"},
	545:           {Sym: "appleqtcsrvr", Description: "appleqtcsrvr"},
	546:           {Sym: "dhcpv6-client", Description: "DHCPv6 Client"},
	547:           {Sym: "dhcpv6-server", Description: "DHCPv6 Server"},
	548:           {Sym: "afpovertcp", Description: "AFP over TCP"},
	549:           {Sym: "idfp", Description: "IDFP"},
	550:           {Sym: "new-rwho", Description: "new-who"},
	551:           {Sym: "cybercash", Description: "cybercash"},
	552:           {Sym: "devshr-nts", Description: "DeviceShare"},
	553:           {Sym: "pirp", Description: "pirp"},
	554:           {Sym: "rtsp", Description: "Real Time Stream Control Protocol"},
	555:           {Sym: "dsf"},
	556:           {Sym: "remotefs", Description: "rfs server"},
	557:           {Sym: "openvms-sysipc", Description: "openvms-sysipc"},
	558:           {Sym: "sdnskmp", Description: "SDNSKMP"},
	559:           {Sym: "teedtap", Description: "TEEDTAP"},
	560:           {Sym: "rmonitor", Description: "rmonitord"},
	561:           {Sym: "monitor"},
	562:           {Sym: "chshell", Description: "chcmd"},
	563:           {Sym: "nntps", Description: "nntp protocol over TLS/SSL (was snntp)"},
	564:           {Sym: "9pfs", Description: "plan 9 file service"},
	565:           {Sym: "whoami", Description: "whoami"},
	566:           {Sym: "streettalk", Description: "streettalk"},
	567:           {Sym: "banyan-rpc", Description: "banyan-rpc"},
	568:           {Sym: "ms-shuttle", Description: "microsoft shuttle"},
	569:           {Sym: "ms-rome", Description: "microsoft rome"},
	570:           {Sym: "meter", Description: "demon"},
	571:           {Sym: "meter", Description: "udemon"},
	572:           {Sym: "sonar", Description: "sonar"},
	573:           {Sym: "banyan-vip", Description: "banyan-vip"},
	574:           {Sym: "ftp-agent", Description: "FTP Software Agent System"},
	575:           {Sym: "vemmi", Description: "VEMMI"},
	576:           {Sym: "ipcd", Description: "ipcd"},
	577:           {Sym: "vnas", Description: "vnas"},
	578:           {Sym: "ipdd", Description: "ipdd"},
	579:           {Sym: "decbsrv", Description: "decbsrv"},
	580:           {Sym: "sntp-heartbeat", Description: "SNTP HEARTBEAT"},
	581:           {Sym: "bdp", Description: "Bundle Discovery Protocol"},
	582:           {Sym: "scc-security", Description: "SCC Security"},
	583:           {Sym: "philips-vc", Description: "Philips Video-Conferencing"},
	584:           {Sym: "keyserver", Description: "Key Server"},
	585:           {Sym: "imap4-ssl", Description: "IMAP4+SSL (use 993 instead)"},
	586:           {Sym: "password-chg", Description: "Password Change"},
	587:           {Sym: "submission", Description: "Submission"},
	588:           {Sym: "cal", Description: "CAL"},
	589:           {Sym: "eyelink", Description: "EyeLink"},
	590:           {Sym: "tns-cml", Description: "TNS CML"},
	591:           {Sym: "http-alt", Description: "FileMaker, Inc. - HTTP Alternate (see Port 80)"},
	592:           {Sym: "eudora-set", Description: "Eudora Set"},
	593:           {Sym: "http-rpc-epmap", Description: "HTTP RPC Ep Map"},
	594:           {Sym: "tpip", Description: "TPIP"},
	595:           {Sym: "cab-protocol", Description: "CAB Protocol"},
	596:           {Sym: "smsd", Description: "SMSD"},
	597:           {Sym: "ptcnameservice", Description: "PTC Name Service"},
	598:           {Sym: "sco-websrvrmg3", Description: "SCO Web Server Manager 3"},
	599:           {Sym: "acp", Description: "Aeolon Core Protocol"},
	600:           {Sym: "ipcserver", Description: "Sun IPC server"},
	601:           {Sym: "syslog-conn", Description: "Reliable Syslog Service"},
	602:           {Sym: "xmlrpc-beep", Description: "XML-RPC over BEEP"},
	603:           {Sym: "idxp", Description: "IDXP"},
	604:           {Sym: "tunnel", Description: "TUNNEL"},
	605:           {Sym: "soap-beep", Description: "SOAP over BEEP"},
	606:           {Sym: "urm", Description: "Cray Unified Resource Manager"},
	607:           {Sym: "nqs", Description: "nqs"},
	608:           {Sym: "sift-uft", Description: "Sender-Initiated/Unsolicited File Transfer"},
	609:           {Sym: "npmp-trap", Description: "npmp-trap"},
	610:           {Sym: "npmp-local", Description: "npmp-local"},
	611:           {Sym: "npmp-gui", Description: "npmp-gui"},
	612:           {Sym: "hmmp-ind", Description: "HMMP Indication"},
	613:           {Sym: "hmmp-op", Description: "HMMP Operation"},
	614:           {Sym: "sshell", Description: "SSLshell"},
	615:           {Sym: "sco-inetmgr", Description: "Internet Configuration Manager"},
	616:           {Sym: "sco-sysmgr", Description: "SCO System Administration Server"},
	617:           {Sym: "sco-dtmgr", Description: "SCO Desktop Administration Server"},
	618:           {Sym: "dei-icda", Description: "DEI-ICDA"},
	619:           {Sym: "compaq-evm", Description: "Compaq EVM"},
	620:           {Sym: "sco-websrvrmgr", Description: "SCO WebServer Manager"},
	621:           {Sym: "escp-ip", Description: "ESCP"},
	622:           {Sym: "collaborator", Description: "Collaborator"},
	623:           {Sym: "asf-rmcp", Description: "ASF Remote Management and Control Protocol"},
	624:           {Sym: "cryptoadmin", Description: "Crypto Admin"},
	625:           {Sym: "dec_dlm", Description: "DEC DLM"},
	626:           {Sym: "asia", Description: "ASIA"},
	627:           {Sym: "passgo-tivoli", Description: "PassGo Tivoli"},
	628:           {Sym: "qmqp", Description: "QMQP"},
	629:           {Sym: "3com-amp3", Description: "3Com AMP3"},
	630:           {Sym: "rda", Description: "RDA"},
	631:           {Sym: "ipp", Description: "IPP (Internet Printing Protocol)"},
	632:           {Sym: "bmpp", Description: "bmpp"},
	633:           {Sym: "servstat", Description: "Service Status update (Sterling Software)"},
	634:           {Sym: "ginad", Description: "ginad"},
	635:           {Sym: "rlzdbase", Description: "RLZ DBase"},
	636:           {Sym: "ldaps", Description: "ldap protocol over TLS/SSL (was sldap)"},
	637:           {Sym: "lanserver", Description: "lanserver"},
	638:           {Sym: "mcns-sec", Description: "mcns-sec"},
	639:           {Sym: "msdp", Description: "MSDP"},
	640:           {Sym: "entrust-sps", Description: "entrust-sps"},
	641:           {Sym: "repcmd", Description: "repcmd"},
	642:           {Sym: "esro-emsdp", Description: "ESRO-EMSDP V1.3"},
	643:           {Sym: "sanity", Description: "SANity"},
	644:           {Sym: "dwr", Description: "dwr"},
	645:           {Sym: "pssc", Description: "PSSC"},
	646:           {Sym: "ldp", Description: "LDP"},
	647:           {Sym: "dhcp-failover", Description: "DHCP Failover"},
	648:           {Sym: "rrp", Description: "Registry Registrar Protocol (RRP)"},
	649:           {Sym: "cadview-3d", Description: "Cadview-3d - streaming 3d models over the internet"},
	650:           {Sym: "obex", Description: "OBEX"},
	651:           {Sym: "ieee-mms", Description: "IEEE MMS"},
	652:           {Sym: "hello-port", Description: "HELLO_PORT\t"},
	653:           {Sym: "repscmd", Description: "RepCmd"},
	654:           {Sym: "aodv", Description: "AODV"},
	655:           {Sym: "tinc", Description: "TINC"},
	656:           {Sym: "spmp", Description: "SPMP"},
	657:           {Sym: "rmc", Description: "RMC"},
	658:           {Sym: "tenfold", Description: "TenFold"},
	660:           {Sym: "mac-srvr-admin", Description: "MacOS Server Admin"},
	661:           {Sym: "hap", Description: "HAP"},
	662:           {Sym: "pftp", Description: "PFTP"},
	663:           {Sym: "purenoise", Description: "PureNoise"},
	664:           {Sym: "asf-secure-rmcp", Description: "ASF Secure Remote Management and Control Protocol"},
	665:           {Sym: "sun-dr", Description: "Sun DR"},
	666:           {Sym: "mdqs"},
	667:           {Sym: "disclose", Description: "campaign contribution disclosures - SDR Technologies"},
	668:           {Sym: "mecomm", Description: "MeComm"},
	669:           {Sym: "meregister", Description: "MeRegister"},
	670:           {Sym: "vacdsm-sws", Description: "VACDSM-SWS"},
	671:           {Sym: "vacdsm-app", Description: "VACDSM-APP"},
	672:           {Sym: "vpps-qua", Description: "VPPS-QUA"},
	673:           {Sym: "cimplex", Description: "CIMPLEX"},
	674:           {Sym: "acap", Description: "ACAP"},
	675:           {Sym: "dctp", Description: "DCTP"},
	676:           {Sym: "vpps-via", Description: "VPPS Via"},
	677:           {Sym: "vpp", Description: "Virtual Presence Protocol"},
	678:           {Sym: "ggf-ncp", Description: "GNU Generation Foundation NCP"},
	679:           {Sym: "mrm", Description: "MRM"},
	680:           {Sym: "entrust-aaas", Description: "entrust-aaas"},
	681:           {Sym: "entrust-aams", Description: "entrust-aams"},
	682:           {Sym: "xfr", Description: "XFR"},
	683:           {Sym: "corba-iiop", Description: "CORBA IIOP"},
	684:           {Sym: "corba-iiop-ssl", Description: "CORBA IIOP SSL"},
	685:           {Sym: "mdc-portmapper", Description: "MDC Port Mapper"},
	686:           {Sym: "hcp-wismar", Description: "Hardware Control Protocol Wismar"},
	687:           {Sym: "asipregistry", Description: "asipregistry"},
	688:           {Sym: "realm-rusd", Description: "REALM-RUSD"},
	689:           {Sym: "nmap", Description: "NMAP"},
	690:           {Sym: "vatp", Description: "VATP"},
	691:           {Sym: "msexch-routing", Description: "MS Exchange Routing"},
	692:           {Sym: "hyperwave-isp", Description: "Hyperwave-ISP"},
	693:           {Sym: "connendp", Description: "connendp"},
	694:           {Sym: "ha-cluster", Description: "ha-cluster"},
	695:           {Sym: "ieee-mms-ssl", Description: "IEEE-MMS-SSL"},
	696:           {Sym: "rushd", Description: "RUSHD"},
	697:           {Sym: "uuidgen", Description: "UUIDGEN"},
	698:           {Sym: "olsr", Description: "OLSR"},
	699:           {Sym: "accessnetwork", Description: "Access Network"},
	700:           {Sym: "epp", Description: "Extensible Provisioning Protocol"},
	701:           {Sym: "lmp", Description: "Link Management Protocol (LMP)"},
	702:           {Sym: "iris-beep", Description: "IRIS over BEEP"},
	704:           {Sym: "elcsd", Description: "errlog copy/server daemon"},
	705:           {Sym: "agentx", Description: "AgentX"},
	706:           {Sym: "silc", Description: "SILC"},
	707:           {Sym: "borland-dsj", Description: "Borland DSJ"},
	709:           {Sym: "entrust-kmsh", Description: "Entrust Key Management Service Handler"},
	710:           {Sym: "entrust-ash", Description: "Entrust Administration Service Handler"},
	711:           {Sym: "cisco-tdp", Description: "Cisco TDP"},
	712:           {Sym: "tbrpf", Description: "TBRPF"},
	729:           {Sym: "netviewdm1", Description: "IBM NetView DM/6000 Server/Client"},
	730:           {Sym: "netviewdm2", Description: "IBM NetView DM/6000 send/tcp"},
	731:           {Sym: "netviewdm3", Description: "IBM NetView DM/6000 receive/tcp"},
	741:           {Sym: "netgw", Description: "netGW"},
	742:           {Sym: "netrcs", Description: "Network based Rev. Cont. Sys"},
	744:           {Sym: "flexlm", Description: "Flexible License Manager"},
	747:           {Sym: "fujitsu-dev", Description: "Fujitsu Device Control"},
	748:           {Sym: "ris-cm", Description: "Russell Info Sci Calendar Manager"},
	749:           {Sym: "kerberos-adm", Description: "kerberos administration"},
	750:           {Sym: "loadav"},
	751:           {Sym: "pump"},
	752:           {Sym: "qrh"},
	753:           {Sym: "rrh"},
	754:           {Sym: "tell", Description: "send"},
	758:           {Sym: "nlogin"},
	759:           {Sym: "con"},
	760:           {Sym: "ns"},
	761:           {Sym: "rxe"},
	762:           {Sym: "quotad"},
	763:           {Sym: "cycleserv"},
	764:           {Sym: "omserv"},
	765:           {Sym: "webster"},
	767:           {Sym: "phonebook", Description: "phone"},
	769:           {Sym: "vid"},
	770:           {Sym: "cadlock"},
	771:           {Sym: "rtip"},
	772:           {Sym: "cycleserv2"},
	773:           {Sym: "notify"},
	774:           {Sym: "acmaint_dbd"},
	775:           {Sym: "acmaint_transd"},
	776:           {Sym: "wpages"},
	777:           {Sym: "multiling-http", Description: "Multiling HTTP"},
	780:           {Sym: "wpgs"},
	800:           {Sym: "mdbs_daemon"},
	801:           {Sym: "device"},
	810:           {Sym: "fcp-udp", Description: "FCP Datagram"},
	828:           {Sym: "itm-mcell-s", Description: "itm-mcell-s"},
	829:           {Sym: "pkix-3-ca-ra", Description: "PKIX-3 CA/RA"},
	830:           {Sym: "netconf-ssh", Description: "NETCONF over SSH"},
	831:           {Sym: "netconf-beep", Description: "NETCONF over BEEP"},
	832:           {Sym: "netconfsoaphttp", Description: "NETCONF for SOAP over HTTPS"},
	833:           {Sym: "netconfsoapbeep", Description: "NETCONF for SOAP over BEEP"},
	847:           {Sym: "dhcp-failover2", Description: "dhcp-failover 2"},
	848:           {Sym: "gdoi", Description: "GDOI"},
	860:           {Sym: "iscsi", Description: "iSCSI"},
	861:           {Sym: "owamp-control", Description: "OWAMP-Control"},
	873:           {Sym: "rsync", Description: "rsync"},
	886:           {Sym: "iclcnet-locate", Description: "ICL coNETion locate server"},
	887:           {Sym: "iclcnet_svinfo", Description: "ICL coNETion server info"},
	888:           {Sym: "accessbuilder", Description: "AccessBuilder"},
	900:           {Sym: "omginitialrefs", Description: "OMG Initial Refs"},
	901:           {Sym: "smpnameres", Description: "SMPNAMERES"},
	902:           {Sym: "ideafarm-chat", Description: "IDEAFARM-CHAT"},
	903:           {Sym: "ideafarm-catch", Description: "IDEAFARM-CATCH"},
	910:           {Sym: "kink", Description: "Kerberized Internet Negotiation of Keys (KINK)"},
	911:           {Sym: "xact-backup", Description: "xact-backup"},
	912:           {Sym: "apex-mesh", Description: "APEX relay-relay service"},
	913:           {Sym: "apex-edge", Description: "APEX endpoint-relay service"},
	989:           {Sym: "ftps-data", Description: "ftp protocol, data, over TLS/SSL"},
	990:           {Sym: "ftps", Description: "ftp protocol, control, over TLS/SSL"},
	991:           {Sym: "nas", Description: "Netnews Administration System"},
	992:           {Sym: "telnets", Description: "telnet protocol over TLS/SSL"},
	993:           {Sym: "imaps", Description: "imap4 protocol over TLS/SSL"},
	994:           {Sym: "ircs", Description: "irc protocol over TLS/SSL"},
	995:           {Sym: "pop3s", Description: "pop3 protocol over TLS/SSL (was spop3)"},
	996:           {Sym: "vsinet", Description: "vsinet"},
	997:           {Sym: "maitrd"},
	998:           {Sym: "puparp"},
	999:           {Sym: "applix", Description: "Applix ac"},
	1000:          {Sym: "cadlock2"},
	1010:          {Sym: "surf", Description: "surf"},

	UDPPortMDNS: {Sym: "mdns", Description: "Multicast DNS"},
}

const (
	TCPPortDomain = 53
	TCPPortRTMP   = 1935
)

var TCPPortMap = scalar.UintMap{
	1:             {Sym: "tcpmux", Description: "TCP Port Service Multiplexer"},
	2:             {Sym: "compressnet", Description: "Management Utility"},
	3:             {Sym: "compressnet", Description: "Compression Process"},
	5:             {Sym: "rje", Description: "Remote Job Entry"},
	7:             {Sym: "echo", Description: "Echo"},
	9:             {Sym: "discard", Description: "Discard"},
	11:            {Sym: "systat", Description: "Active Users"},
	13:            {Sym: "daytime", Description: "Daytime (RFC 867)"},
	17:            {Sym: "qotd", Description: "Quote of the Day"},
	18:            {Sym: "msp", Description: "Message Send Protocol"},
	19:            {Sym: "chargen", Description: "Character Generator"},
	20:            {Sym: "ftp-data", Description: "File Transfer [Default Data]"},
	21:            {Sym: "ftp", Description: "File Transfer [Control]"},
	22:            {Sym: "ssh", Description: "SSH Remote Login Protocol"},
	23:            {Sym: "telnet", Description: "Telnet"},
	25:            {Sym: "smtp", Description: "Simple Mail Transfer"},
	27:            {Sym: "nsw-fe", Description: "NSW User System FE"},
	29:            {Sym: "msg-icp", Description: "MSG ICP"},
	31:            {Sym: "msg-auth", Description: "MSG Authentication"},
	33:            {Sym: "dsp", Description: "Display Support Protocol"},
	37:            {Sym: "time", Description: "Time"},
	38:            {Sym: "rap", Description: "Route Access Protocol"},
	39:            {Sym: "rlp", Description: "Resource Location Protocol"},
	41:            {Sym: "graphics", Description: "Graphics"},
	42:            {Sym: "name", Description: "Host Name Server"},
	44:            {Sym: "mpm-flags", Description: "MPM FLAGS Protocol"},
	45:            {Sym: "mpm", Description: "Message Processing Module [recv]"},
	46:            {Sym: "mpm-snd", Description: "MPM [default send]"},
	47:            {Sym: "ni-ftp", Description: "NI FTP"},
	48:            {Sym: "auditd", Description: "Digital Audit Daemon"},
	49:            {Sym: "tacacs", Description: "Login Host Protocol (TACACS)"},
	50:            {Sym: "re-mail-ck", Description: "Remote Mail Checking Protocol"},
	51:            {Sym: "la-maint", Description: "IMP Logical Address Maintenance"},
	52:            {Sym: "xns-time", Description: "XNS Time Protocol"},
	TCPPortDomain: {Sym: "domain", Description: "Domain Name Server"},
	54:            {Sym: "xns-ch", Description: "XNS Clearinghouse"},
	55:            {Sym: "isi-gl", Description: "ISI Graphics Language"},
	56:            {Sym: "xns-auth", Description: "XNS Authentication"},
	58:            {Sym: "xns-mail", Description: "XNS Mail"},
	61:            {Sym: "ni-mail", Description: "NI MAIL"},
	62:            {Sym: "acas", Description: "ACA Services"},
	64:            {Sym: "covia", Description: "Communications Integrator (CI)"},
	65:            {Sym: "tacacs-ds", Description: "TACACS-Database Service"},
	66:            {Sym: "net", Description: "Oracle SQL*NET"},
	67:            {Sym: "bootps", Description: "Bootstrap Protocol Server"},
	68:            {Sym: "bootpc", Description: "Bootstrap Protocol Client"},
	69:            {Sym: "tftp", Description: "Trivial File Transfer"},
	70:            {Sym: "gopher", Description: "Gopher"},
	71:            {Sym: "netrjs-1", Description: "Remote Job Service"},
	72:            {Sym: "netrjs-2", Description: "Remote Job Service"},
	73:            {Sym: "netrjs-3", Description: "Remote Job Service"},
	74:            {Sym: "netrjs-4", Description: "Remote Job Service"},
	76:            {Sym: "deos", Description: "Distributed External Object Store"},
	78:            {Sym: "vettcp", Description: "vettcp"},
	79:            {Sym: "finger", Description: "Finger"},
	80:            {Sym: "http", Description: "World Wide Web HTTP"},
	81:            {Sym: "hosts2-ns", Description: "HOSTS2 Name Server"},
	82:            {Sym: "xfer", Description: "XFER Utility"},
	83:            {Sym: "mit-ml-dev", Description: "MIT ML Device"},
	84:            {Sym: "ctf", Description: "Common Trace Facility"},
	85:            {Sym: "mit-ml-dev", Description: "MIT ML Device"},
	86:            {Sym: "mfcobol", Description: "Micro Focus Cobol"},
	88:            {Sym: "kerberos", Description: "Kerberos"},
	89:            {Sym: "su-mit-tg", Description: "SU/MIT Telnet Gateway"},
	90:            {Sym: "dnsix", Description: "DNSIX Securit Attribute Token Map"},
	91:            {Sym: "mit-dov", Description: "MIT Dover Spooler"},
	92:            {Sym: "npp", Description: "Network Printing Protocol"},
	93:            {Sym: "dcp", Description: "Device Control Protocol"},
	94:            {Sym: "objcall", Description: "Tivoli Object Dispatcher"},
	95:            {Sym: "supdup", Description: "SUPDUP"},
	96:            {Sym: "dixie", Description: "DIXIE Protocol Specification"},
	97:            {Sym: "swift-rvf", Description: "Swift Remote Virtural File Protocol"},
	98:            {Sym: "tacnews", Description: "TAC News"},
	99:            {Sym: "metagram", Description: "Metagram Relay"},
	100:           {Sym: "newacct", Description: "[unauthorized use]"},
	101:           {Sym: "hostname", Description: "NIC Host Name Server"},
	102:           {Sym: "iso-tsap", Description: "ISO-TSAP Class 0"},
	103:           {Sym: "gppitnp", Description: "Genesis Point-to-Point Trans Net"},
	104:           {Sym: "acr-nema", Description: "ACR-NEMA Digital Imag. & Comm. 300"},
	105:           {Sym: "cso", Description: "CCSO name server protocol"},
	106:           {Sym: "3com-tsmux", Description: "3COM-TSMUX"},
	107:           {Sym: "rtelnet", Description: "Remote Telnet Service"},
	108:           {Sym: "snagas", Description: "SNA Gateway Access Server"},
	109:           {Sym: "pop2", Description: "Post Office Protocol - Version 2"},
	110:           {Sym: "pop3", Description: "Post Office Protocol - Version 3"},
	111:           {Sym: "sunrpc", Description: "SUN Remote Procedure Call"},
	112:           {Sym: "mcidas", Description: "McIDAS Data Transmission Protocol"},
	113:           {Sym: "auth"},
	114:           {Sym: "audionews", Description: "Audio News Multicast"},
	115:           {Sym: "sftp", Description: "Simple File Transfer Protocol"},
	116:           {Sym: "ansanotify", Description: "ANSA REX Notify"},
	117:           {Sym: "uucp-path", Description: "UUCP Path Service"},
	118:           {Sym: "sqlserv", Description: "SQL Services"},
	119:           {Sym: "nntp", Description: "Network News Transfer Protocol"},
	120:           {Sym: "cfdptkt", Description: "CFDPTKT"},
	121:           {Sym: "erpc", Description: "Encore Expedited Remote Pro.Call"},
	122:           {Sym: "smakynet", Description: "SMAKYNET"},
	123:           {Sym: "ntp", Description: "Network Time Protocol"},
	124:           {Sym: "ansatrader", Description: "ANSA REX Trader"},
	125:           {Sym: "locus-map", Description: "Locus PC-Interface Net Map Ser"},
	126:           {Sym: "nxedit", Description: "NXEdit"},
	127:           {Sym: "locus-con", Description: "Locus PC-Interface Conn Server"},
	128:           {Sym: "gss-xlicen", Description: "GSS X License Verification"},
	129:           {Sym: "pwdgen", Description: "Password Generator Protocol"},
	130:           {Sym: "cisco-fna", Description: "cisco FNATIVE"},
	131:           {Sym: "cisco-tna", Description: "cisco TNATIVE"},
	132:           {Sym: "cisco-sys", Description: "cisco SYSMAINT"},
	133:           {Sym: "statsrv", Description: "Statistics Service"},
	134:           {Sym: "ingres-net", Description: "INGRES-NET Service"},
	135:           {Sym: "epmap", Description: "DCE endpoint resolution"},
	136:           {Sym: "profile", Description: "PROFILE Naming System"},
	137:           {Sym: "netbios-ns", Description: "NETBIOS Name Service"},
	138:           {Sym: "netbios-dgm", Description: "NETBIOS Datagram Service"},
	139:           {Sym: "netbios-ssn", Description: "NETBIOS Session Service"},
	140:           {Sym: "emfis-data", Description: "EMFIS Data Service"},
	141:           {Sym: "emfis-cntl", Description: "EMFIS Control Service"},
	142:           {Sym: "bl-idm", Description: "Britton-Lee IDM"},
	143:           {Sym: "imap", Description: "Internet Message Access Protocol"},
	144:           {Sym: "uma", Description: "Universal Management Architecture"},
	145:           {Sym: "uaac", Description: "UAAC Protocol"},
	146:           {Sym: "iso-tp0", Description: "ISO-IP0"},
	147:           {Sym: "iso-ip", Description: "ISO-IP"},
	148:           {Sym: "jargon", Description: "Jargon"},
	149:           {Sym: "aed-512", Description: "AED 512 Emulation Service"},
	150:           {Sym: "sql-net", Description: "SQL-NET"},
	151:           {Sym: "hems", Description: "HEMS"},
	152:           {Sym: "bftp", Description: "Background File Transfer Program"},
	153:           {Sym: "sgmp", Description: "SGMP"},
	154:           {Sym: "netsc-prod", Description: "NETSC"},
	155:           {Sym: "netsc-dev", Description: "NETSC"},
	156:           {Sym: "sqlsrv", Description: "SQL Service"},
	157:           {Sym: "knet-cmp", Description: "KNET/VM Command/Message Protocol"},
	158:           {Sym: "pcmail-srv", Description: "PCMail Server"},
	159:           {Sym: "nss-routing", Description: "NSS-Routing"},
	160:           {Sym: "sgmp-traps", Description: "SGMP-TRAPS"},
	161:           {Sym: "snmp", Description: "SNMP"},
	162:           {Sym: "snmptrap", Description: "SNMPTRAP"},
	163:           {Sym: "cmip-man", Description: "CMIP/TCP Manager"},
	164:           {Sym: "cmip-agent", Description: "CMIP/TCP Agent"},
	165:           {Sym: "xns-courier", Description: "Xerox"},
	166:           {Sym: "s-net", Description: "Sirius Systems"},
	167:           {Sym: "namp", Description: "NAMP"},
	168:           {Sym: "rsvd", Description: "RSVD"},
	169:           {Sym: "send", Description: "SEND"},
	170:           {Sym: "print-srv", Description: "Network PostScript"},
	171:           {Sym: "multiplex", Description: "Network Innovations Multiplex"},
	172:           {Sym: "1", Description: "Network Innovations CL/1"},
	173:           {Sym: "xyplex-mux", Description: "Xyplex"},
	174:           {Sym: "mailq", Description: "MAILQ"},
	175:           {Sym: "vmnet", Description: "VMNET"},
	176:           {Sym: "genrad-mux", Description: "GENRAD-MUX"},
	177:           {Sym: "xdmcp", Description: "X Display Manager Control Protocol"},
	178:           {Sym: "nextstep", Description: "NextStep Window Server"},
	179:           {Sym: "bgp", Description: "Border Gateway Protocol"},
	180:           {Sym: "ris", Description: "Intergraph"},
	181:           {Sym: "unify", Description: "Unify"},
	182:           {Sym: "audit", Description: "Unisys Audit SITP"},
	183:           {Sym: "ocbinder", Description: "OCBinder"},
	184:           {Sym: "ocserver", Description: "OCServer"},
	185:           {Sym: "remote-kis", Description: "Remote-KIS"},
	186:           {Sym: "kis", Description: "KIS Protocol"},
	187:           {Sym: "aci", Description: "Application Communication Interface"},
	188:           {Sym: "mumps", Description: "Plus Five's MUMPS"},
	189:           {Sym: "qft", Description: "Queued File Transport"},
	190:           {Sym: "gacp", Description: "Gateway Access Control Protocol"},
	191:           {Sym: "prospero", Description: "Prospero Directory Service"},
	192:           {Sym: "osu-nms", Description: "OSU Network Monitoring System"},
	193:           {Sym: "srmp", Description: "Spider Remote Monitoring Protocol"},
	194:           {Sym: "irc", Description: "Internet Relay Chat Protocol"},
	195:           {Sym: "dn6-nlm-aud", Description: "DNSIX Network Level Module Audit"},
	196:           {Sym: "dn6-smm-red", Description: "DNSIX Session Mgt Module Audit Redir"},
	197:           {Sym: "dls", Description: "Directory Location Service"},
	198:           {Sym: "dls-mon", Description: "Directory Location Service Monitor"},
	199:           {Sym: "smux", Description: "SMUX"},
	200:           {Sym: "src", Description: "IBM System Resource Controller"},
	201:           {Sym: "at-rtmp", Description: "AppleTalk Routing Maintenance"},
	202:           {Sym: "at-nbp", Description: "AppleTalk Name Binding"},
	203:           {Sym: "at-3", Description: "AppleTalk Unused"},
	204:           {Sym: "at-echo", Description: "AppleTalk Echo"},
	205:           {Sym: "at-5", Description: "AppleTalk Unused"},
	206:           {Sym: "at-zis", Description: "AppleTalk Zone Information"},
	207:           {Sym: "at-7", Description: "AppleTalk Unused"},
	208:           {Sym: "at-8", Description: "AppleTalk Unused"},
	209:           {Sym: "qmtp", Description: "The Quick Mail Transfer Protocol"},
	210:           {Sym: "50", Description: "ANSI Z39.50"},
	211:           {Sym: "g", Description: "Texas Instruments 914C/G Terminal"},
	212:           {Sym: "anet", Description: "ATEXSSTR"},
	213:           {Sym: "ipx", Description: "IPX         \t"},
	214:           {Sym: "vmpwscs", Description: "VM PWSCS"},
	215:           {Sym: "softpc", Description: "Insignia Solutions"},
	216:           {Sym: "cailic", Description: "Computer Associates Int'l License Server"},
	217:           {Sym: "dbase", Description: "dBASE Unix"},
	218:           {Sym: "mpp", Description: "Netix Message Posting Protocol"},
	219:           {Sym: "uarps", Description: "Unisys ARPs"},
	220:           {Sym: "imap3", Description: "Interactive Mail Access Protocol v3"},
	221:           {Sym: "fln-spx", Description: "Berkeley rlogind with SPX auth"},
	222:           {Sym: "rsh-spx", Description: "Berkeley rshd with SPX auth"},
	223:           {Sym: "cdc", Description: "Certificate Distribution Center"},
	224:           {Sym: "masqdialer", Description: "masqdialer"},
	242:           {Sym: "direct", Description: "Direct"},
	243:           {Sym: "sur-meas", Description: "Survey Measurement"},
	244:           {Sym: "inbusiness", Description: "inbusiness"},
	245:           {Sym: "link", Description: "LINK"},
	246:           {Sym: "dsp3270", Description: "Display Systems Protocol"},
	247:           {Sym: "subntbcst_tftp", Description: "SUBNTBCST_TFTP"},
	248:           {Sym: "bhfhs", Description: "bhfhs"},
	256:           {Sym: "rap", Description: "RAP"},
	257:           {Sym: "set", Description: "Secure Electronic Transaction"},
	258:           {Sym: "yak-chat", Description: "Yak Winsock Personal Chat"},
	259:           {Sym: "esro-gen", Description: "Efficient Short Remote Operations"},
	260:           {Sym: "openport", Description: "Openport"},
	261:           {Sym: "nsiiops", Description: "IIOP Name Service over TLS/SSL"},
	262:           {Sym: "arcisdms", Description: "Arcisdms"},
	263:           {Sym: "hdap", Description: "HDAP"},
	264:           {Sym: "bgmp", Description: "BGMP"},
	265:           {Sym: "x-bone-ctl", Description: "X-Bone CTL"},
	266:           {Sym: "sst", Description: "SCSI on ST"},
	267:           {Sym: "td-service", Description: "Tobit David Service Layer"},
	268:           {Sym: "td-replica", Description: "Tobit David Replica"},
	280:           {Sym: "http-mgmt", Description: "http-mgmt"},
	281:           {Sym: "personal-link", Description: "Personal Link"},
	282:           {Sym: "cableport-ax", Description: "Cable Port A/X"},
	283:           {Sym: "rescap", Description: "rescap"},
	284:           {Sym: "corerjd", Description: "corerjd"},
	286:           {Sym: "fxp-1", Description: "FXP-1"},
	287:           {Sym: "k-block", Description: "K-BLOCK"},
	308:           {Sym: "novastorbakcup", Description: "Novastor Backup"},
	309:           {Sym: "entrusttime", Description: "EntrustTime"},
	310:           {Sym: "bhmds", Description: "bhmds"},
	311:           {Sym: "asip-webadmin", Description: "AppleShare IP WebAdmin"},
	312:           {Sym: "vslmp", Description: "VSLMP"},
	313:           {Sym: "magenta-logic", Description: "Magenta Logic"},
	314:           {Sym: "opalis-robot", Description: "Opalis Robot"},
	315:           {Sym: "dpsi", Description: "DPSI"},
	316:           {Sym: "decauth", Description: "decAuth"},
	317:           {Sym: "zannet", Description: "Zannet"},
	318:           {Sym: "pkix-timestamp", Description: "PKIX TimeStamp"},
	319:           {Sym: "ptp-event", Description: "PTP Event"},
	320:           {Sym: "ptp-general", Description: "PTP General"},
	321:           {Sym: "pip", Description: "PIP"},
	322:           {Sym: "rtsps", Description: "RTSPS"},
	333:           {Sym: "texar", Description: "Texar Security Port"},
	344:           {Sym: "pdap", Description: "Prospero Data Access Protocol"},
	345:           {Sym: "pawserv", Description: "Perf Analysis Workbench"},
	346:           {Sym: "zserv", Description: "Zebra server"},
	347:           {Sym: "fatserv", Description: "Fatmen Server"},
	348:           {Sym: "csi-sgwp", Description: "Cabletron Management Protocol"},
	349:           {Sym: "mftp", Description: "mftp"},
	350:           {Sym: "matip-type-a", Description: "MATIP Type A"},
	351:           {Sym: "matip-type-b", Description: "MATIP Type B"},
	352:           {Sym: "dtag-ste-sb", Description: "DTAG (assigned long ago)"},
	353:           {Sym: "ndsauth", Description: "NDSAUTH"},
	354:           {Sym: "bh611", Description: "bh611"},
	355:           {Sym: "datex-asn", Description: "DATEX-ASN"},
	356:           {Sym: "cloanto-net-1", Description: "Cloanto Net 1"},
	357:           {Sym: "bhevent", Description: "bhevent"},
	358:           {Sym: "shrinkwrap", Description: "Shrinkwrap"},
	359:           {Sym: "nsrmp", Description: "Network Security Risk Management Protocol"},
	360:           {Sym: "scoi2odialog", Description: "scoi2odialog"},
	361:           {Sym: "semantix", Description: "Semantix"},
	362:           {Sym: "srssend", Description: "SRS Send"},
	363:           {Sym: "rsvp_tunnel", Description: "RSVP Tunnel"},
	364:           {Sym: "aurora-cmgr", Description: "Aurora CMGR"},
	365:           {Sym: "dtk", Description: "DTK"},
	366:           {Sym: "odmr", Description: "ODMR"},
	367:           {Sym: "mortgageware", Description: "MortgageWare"},
	368:           {Sym: "qbikgdp", Description: "QbikGDP"},
	369:           {Sym: "rpc2portmap", Description: "rpc2portmap"},
	370:           {Sym: "codaauth2", Description: "codaauth2"},
	371:           {Sym: "clearcase", Description: "Clearcase"},
	372:           {Sym: "ulistproc", Description: "ListProcessor"},
	373:           {Sym: "legent-1", Description: "Legent Corporation"},
	374:           {Sym: "legent-2", Description: "Legent Corporation"},
	375:           {Sym: "hassle", Description: "Hassle"},
	376:           {Sym: "nip", Description: "Amiga Envoy Network Inquiry Proto"},
	377:           {Sym: "tnETOS", Description: "NEC Corporation"},
	378:           {Sym: "dsETOS", Description: "NEC Corporation"},
	379:           {Sym: "is99c", Description: "TIA/EIA/IS-99 modem client"},
	380:           {Sym: "is99s", Description: "TIA/EIA/IS-99 modem server"},
	381:           {Sym: "hp-collector", Description: "hp performance data collector"},
	382:           {Sym: "hp-managed-node", Description: "hp performance data managed node"},
	383:           {Sym: "hp-alarm-mgr", Description: "hp performance data alarm manager"},
	384:           {Sym: "arns", Description: "A Remote Network Server System"},
	385:           {Sym: "ibm-app", Description: "IBM Application"},
	386:           {Sym: "asa", Description: "ASA Message Router Object Def"},
	387:           {Sym: "aurp", Description: "Appletalk Update-Based Routing Pro"},
	388:           {Sym: "unidata-ldm", Description: "Unidata LDM"},
	389:           {Sym: "ldap", Description: "Lightweight Directory Access Protocol"},
	390:           {Sym: "uis", Description: "UIS"},
	391:           {Sym: "synotics-relay", Description: "SynOptics SNMP Relay Port"},
	392:           {Sym: "synotics-broker", Description: "SynOptics Port Broker Port"},
	393:           {Sym: "meta5", Description: "Meta5"},
	394:           {Sym: "embl-ndt", Description: "EMBL Nucleic Data Transfer"},
	395:           {Sym: "netcp", Description: "NETscout Control Protocol"},
	396:           {Sym: "netware-ip", Description: "Novell Netware over IP"},
	397:           {Sym: "mptn", Description: "Multi Protocol Trans. Net"},
	398:           {Sym: "kryptolan", Description: "Kryptolan"},
	399:           {Sym: "iso-tsap-c2", Description: "ISO Transport Class 2 Non-Control over TCP"},
	400:           {Sym: "work-sol", Description: "Workstation Solutions"},
	401:           {Sym: "ups", Description: "Uninterruptible Power Supply"},
	402:           {Sym: "genie", Description: "Genie Protocol"},
	403:           {Sym: "decap", Description: "decap"},
	404:           {Sym: "nced", Description: "nced"},
	405:           {Sym: "ncld", Description: "ncld"},
	406:           {Sym: "imsp", Description: "Interactive Mail Support Protocol"},
	407:           {Sym: "timbuktu", Description: "Timbuktu"},
	408:           {Sym: "prm-sm", Description: "Prospero Resource Manager Sys. Man"},
	409:           {Sym: "prm-nm", Description: "Prospero Resource Manager Node Man"},
	410:           {Sym: "decladebug", Description: "DECLadebug Remote Debug Protocol"},
	411:           {Sym: "rmt", Description: "Remote MT Protocol"},
	412:           {Sym: "synoptics-trap", Description: "Trap Convention Port"},
	413:           {Sym: "smsp", Description: "Storage Management Services Protocol"},
	414:           {Sym: "infoseek", Description: "InfoSeek"},
	415:           {Sym: "bnet", Description: "BNet"},
	416:           {Sym: "silverplatter", Description: "Silverplatter"},
	417:           {Sym: "onmux", Description: "Onmux"},
	418:           {Sym: "hyper-g", Description: "Hyper-G"},
	419:           {Sym: "ariel1", Description: "Ariel 1"},
	420:           {Sym: "smpte", Description: "SMPTE"},
	421:           {Sym: "ariel2", Description: "Ariel 2"},
	422:           {Sym: "ariel3", Description: "Ariel 3"},
	423:           {Sym: "opc-job-start", Description: "IBM Operations Planning and Control Start"},
	424:           {Sym: "opc-job-track", Description: "IBM Operations Planning and Control Track"},
	425:           {Sym: "icad-el", Description: "ICAD"},
	426:           {Sym: "smartsdp", Description: "smartsdp"},
	427:           {Sym: "svrloc", Description: "Server Location"},
	428:           {Sym: "ocs_cmu", Description: "OCS_CMU"},
	429:           {Sym: "ocs_amu", Description: "OCS_AMU"},
	430:           {Sym: "utmpsd", Description: "UTMPSD"},
	431:           {Sym: "utmpcd", Description: "UTMPCD"},
	432:           {Sym: "iasd", Description: "IASD"},
	433:           {Sym: "nnsp", Description: "NNSP"},
	434:           {Sym: "mobileip-agent", Description: "MobileIP-Agent"},
	435:           {Sym: "mobilip-mn", Description: "MobilIP-MN"},
	436:           {Sym: "dna-cml", Description: "DNA-CML"},
	437:           {Sym: "comscm", Description: "comscm"},
	438:           {Sym: "dsfgw", Description: "dsfgw"},
	439:           {Sym: "dasp", Description: "dasp      Thomas Obermair"},
	440:           {Sym: "sgcp", Description: "sgcp"},
	441:           {Sym: "decvms-sysmgt", Description: "decvms-sysmgt"},
	442:           {Sym: "cvc_hostd", Description: "cvc_hostd"},
	443:           {Sym: "https", Description: "http protocol over TLS/SSL"},
	444:           {Sym: "snpp", Description: "Simple Network Paging Protocol"},
	445:           {Sym: "microsoft-ds", Description: "Microsoft-DS"},
	446:           {Sym: "ddm-rdb", Description: "DDM-RDB"},
	447:           {Sym: "ddm-dfm", Description: "DDM-RFM"},
	448:           {Sym: "ddm-ssl", Description: "DDM-SSL"},
	449:           {Sym: "as-servermap", Description: "AS Server Mapper"},
	450:           {Sym: "tserver", Description: "Computer Supported Telecomunication Applications"},
	451:           {Sym: "sfs-smp-net", Description: "Cray Network Semaphore server"},
	452:           {Sym: "sfs-config", Description: "Cray SFS config server"},
	453:           {Sym: "creativeserver", Description: "CreativeServer"},
	454:           {Sym: "contentserver", Description: "ContentServer"},
	455:           {Sym: "creativepartnr", Description: "CreativePartnr"},
	456:           {Sym: "macon-tcp", Description: "macon-tcp"},
	457:           {Sym: "scohelp", Description: "scohelp"},
	458:           {Sym: "appleqtc", Description: "apple quick time"},
	459:           {Sym: "ampr-rcmd", Description: "ampr-rcmd"},
	460:           {Sym: "skronk", Description: "skronk"},
	461:           {Sym: "datasurfsrv", Description: "DataRampSrv"},
	462:           {Sym: "datasurfsrvsec", Description: "DataRampSrvSec"},
	463:           {Sym: "alpes", Description: "alpes"},
	464:           {Sym: "kpasswd", Description: "kpasswd"},
	465:           {Sym: "urd", Description: "URL Rendesvous Directory for SSM"},
	466:           {Sym: "digital-vrc", Description: "digital-vrc"},
	467:           {Sym: "mylex-mapd", Description: "mylex-mapd"},
	468:           {Sym: "photuris", Description: "proturis"},
	469:           {Sym: "rcp", Description: "Radio Control Protocol"},
	470:           {Sym: "scx-proxy", Description: "scx-proxy"},
	471:           {Sym: "mondex", Description: "Mondex"},
	472:           {Sym: "ljk-login", Description: "ljk-login"},
	473:           {Sym: "hybrid-pop", Description: "hybrid-pop"},
	474:           {Sym: "tn-tl-w1", Description: "tn-tl-w1"},
	475:           {Sym: "tcpnethaspsrv", Description: "tcpnethaspsrv"},
	476:           {Sym: "tn-tl-fd1", Description: "tn-tl-fd1"},
	477:           {Sym: "ss7ns", Description: "ss7ns"},
	478:           {Sym: "spsc", Description: "spsc"},
	479:           {Sym: "iafserver", Description: "iafserver"},
	480:           {Sym: "iafdbase", Description: "iafdbase"},
	481:           {Sym: "ph", Description: "Ph service"},
	482:           {Sym: "bgs-nsi", Description: "bgs-nsi"},
	483:           {Sym: "ulpnet", Description: "ulpnet"},
	484:           {Sym: "integra-sme", Description: "Integra Software Management Environment"},
	485:           {Sym: "powerburst", Description: "Air Soft Power Burst"},
	486:           {Sym: "avian", Description: "avian"},
	487:           {Sym: "saft", Description: "saft Simple Asynchronous File Transfer"},
	488:           {Sym: "gss-http", Description: "gss-http"},
	489:           {Sym: "nest-protocol", Description: "nest-protocol"},
	490:           {Sym: "micom-pfs", Description: "micom-pfs"},
	491:           {Sym: "go-login", Description: "go-login"},
	492:           {Sym: "ticf-1", Description: "Transport Independent Convergence for FNA"},
	493:           {Sym: "ticf-2", Description: "Transport Independent Convergence for FNA"},
	494:           {Sym: "pov-ray", Description: "POV-Ray"},
	495:           {Sym: "intecourier", Description: "intecourier"},
	496:           {Sym: "pim-rp-disc", Description: "PIM-RP-DISC"},
	497:           {Sym: "dantz", Description: "dantz"},
	498:           {Sym: "siam", Description: "siam"},
	499:           {Sym: "iso-ill", Description: "ISO ILL Protocol"},
	500:           {Sym: "isakmp", Description: "isakmp"},
	501:           {Sym: "stmf", Description: "STMF"},
	502:           {Sym: "asa-appl-proto", Description: "asa-appl-proto"},
	503:           {Sym: "intrinsa", Description: "Intrinsa"},
	504:           {Sym: "citadel", Description: "citadel"},
	505:           {Sym: "mailbox-lm", Description: "mailbox-lm"},
	506:           {Sym: "ohimsrv", Description: "ohimsrv"},
	507:           {Sym: "crs", Description: "crs"},
	508:           {Sym: "xvttp", Description: "xvttp"},
	509:           {Sym: "snare", Description: "snare"},
	510:           {Sym: "fcp", Description: "FirstClass Protocol"},
	511:           {Sym: "passgo", Description: "PassGo"},
	512:           {Sym: "exec", Description: "remote process execution;"},
	513:           {Sym: "login", Description: "remote login a la telnet;"},
	514:           {Sym: "shell", Description: "cmd"},
	515:           {Sym: "printer", Description: "spooler"},
	516:           {Sym: "videotex", Description: "videotex"},
	517:           {Sym: "talk", Description: "like tenex link, but across"},
	518:           {Sym: "ntalk"},
	519:           {Sym: "utime", Description: "unixtime"},
	520:           {Sym: "efs", Description: "extended file name server"},
	521:           {Sym: "ripng", Description: "ripng"},
	522:           {Sym: "ulp", Description: "ULP"},
	523:           {Sym: "ibm-db2", Description: "IBM-DB2"},
	524:           {Sym: "ncp", Description: "NCP"},
	525:           {Sym: "timed", Description: "timeserver"},
	526:           {Sym: "tempo", Description: "newdate"},
	527:           {Sym: "stx", Description: "Stock IXChange"},
	528:           {Sym: "custix", Description: "Customer IXChange"},
	529:           {Sym: "irc-serv", Description: "IRC-SERV"},
	530:           {Sym: "courier", Description: "rpc"},
	531:           {Sym: "conference", Description: "chat"},
	532:           {Sym: "netnews", Description: "readnews"},
	533:           {Sym: "netwall", Description: "for emergency broadcasts"},
	534:           {Sym: "mm-admin", Description: "MegaMedia Admin"},
	535:           {Sym: "iiop", Description: "iiop"},
	536:           {Sym: "opalis-rdv", Description: "opalis-rdv"},
	537:           {Sym: "nmsp", Description: "Networked Media Streaming Protocol"},
	538:           {Sym: "gdomap", Description: "gdomap"},
	539:           {Sym: "apertus-ldp", Description: "Apertus Technologies Load Determination"},
	540:           {Sym: "uucp", Description: "uucpd\t\t"},
	541:           {Sym: "uucp-rlogin", Description: "uucp-rlogin"},
	542:           {Sym: "commerce", Description: "commerce"},
	543:           {Sym: "klogin"},
	544:           {Sym: "kshell", Description: "krcmd"},
	545:           {Sym: "appleqtcsrvr", Description: "appleqtcsrvr"},
	546:           {Sym: "dhcpv6-client", Description: "DHCPv6 Client"},
	547:           {Sym: "dhcpv6-server", Description: "DHCPv6 Server"},
	548:           {Sym: "afpovertcp", Description: "AFP over TCP"},
	549:           {Sym: "idfp", Description: "IDFP"},
	550:           {Sym: "new-rwho", Description: "new-who"},
	551:           {Sym: "cybercash", Description: "cybercash"},
	552:           {Sym: "devshr-nts", Description: "DeviceShare"},
	553:           {Sym: "pirp", Description: "pirp"},
	554:           {Sym: "rtsp", Description: "Real Time Stream Control Protocol"},
	555:           {Sym: "dsf"},
	556:           {Sym: "remotefs", Description: "rfs server"},
	557:           {Sym: "openvms-sysipc", Description: "openvms-sysipc"},
	558:           {Sym: "sdnskmp", Description: "SDNSKMP"},
	559:           {Sym: "teedtap", Description: "TEEDTAP"},
	560:           {Sym: "rmonitor", Description: "rmonitord"},
	561:           {Sym: "monitor"},
	562:           {Sym: "chshell", Description: "chcmd"},
	563:           {Sym: "nntps", Description: "nntp protocol over TLS/SSL (was snntp)"},
	564:           {Sym: "9pfs", Description: "plan 9 file service"},
	565:           {Sym: "whoami", Description: "whoami"},
	566:           {Sym: "streettalk", Description: "streettalk"},
	567:           {Sym: "banyan-rpc", Description: "banyan-rpc"},
	568:           {Sym: "ms-shuttle", Description: "microsoft shuttle"},
	569:           {Sym: "ms-rome", Description: "microsoft rome"},
	570:           {Sym: "meter", Description: "demon"},
	571:           {Sym: "meter", Description: "udemon"},
	572:           {Sym: "sonar", Description: "sonar"},
	573:           {Sym: "banyan-vip", Description: "banyan-vip"},
	574:           {Sym: "ftp-agent", Description: "FTP Software Agent System"},
	575:           {Sym: "vemmi", Description: "VEMMI"},
	576:           {Sym: "ipcd", Description: "ipcd"},
	577:           {Sym: "vnas", Description: "vnas"},
	578:           {Sym: "ipdd", Description: "ipdd"},
	579:           {Sym: "decbsrv", Description: "decbsrv"},
	580:           {Sym: "sntp-heartbeat", Description: "SNTP HEARTBEAT"},
	581:           {Sym: "bdp", Description: "Bundle Discovery Protocol"},
	582:           {Sym: "scc-security", Description: "SCC Security"},
	583:           {Sym: "philips-vc", Description: "Philips Video-Conferencing"},
	584:           {Sym: "keyserver", Description: "Key Server"},
	585:           {Sym: "imap4-ssl", Description: "IMAP4+SSL (use 993 instead)"},
	586:           {Sym: "password-chg", Description: "Password Change"},
	587:           {Sym: "submission", Description: "Submission"},
	588:           {Sym: "cal", Description: "CAL"},
	589:           {Sym: "eyelink", Description: "EyeLink"},
	590:           {Sym: "tns-cml", Description: "TNS CML"},
	591:           {Sym: "http-alt", Description: "FileMaker, Inc. - HTTP Alternate (see Port 80)"},
	592:           {Sym: "eudora-set", Description: "Eudora Set"},
	593:           {Sym: "http-rpc-epmap", Description: "HTTP RPC Ep Map"},
	594:           {Sym: "tpip", Description: "TPIP"},
	595:           {Sym: "cab-protocol", Description: "CAB Protocol"},
	596:           {Sym: "smsd", Description: "SMSD"},
	597:           {Sym: "ptcnameservice", Description: "PTC Name Service"},
	598:           {Sym: "sco-websrvrmg3", Description: "SCO Web Server Manager 3"},
	599:           {Sym: "acp", Description: "Aeolon Core Protocol"},
	600:           {Sym: "ipcserver", Description: "Sun IPC server"},
	601:           {Sym: "syslog-conn", Description: "Reliable Syslog Service"},
	602:           {Sym: "xmlrpc-beep", Description: "XML-RPC over BEEP"},
	603:           {Sym: "idxp", Description: "IDXP"},
	604:           {Sym: "tunnel", Description: "TUNNEL"},
	605:           {Sym: "soap-beep", Description: "SOAP over BEEP"},
	606:           {Sym: "urm", Description: "Cray Unified Resource Manager"},
	607:           {Sym: "nqs", Description: "nqs"},
	608:           {Sym: "sift-uft", Description: "Sender-Initiated/Unsolicited File Transfer"},
	609:           {Sym: "npmp-trap", Description: "npmp-trap"},
	610:           {Sym: "npmp-local", Description: "npmp-local"},
	611:           {Sym: "npmp-gui", Description: "npmp-gui"},
	612:           {Sym: "hmmp-ind", Description: "HMMP Indication"},
	613:           {Sym: "hmmp-op", Description: "HMMP Operation"},
	614:           {Sym: "sshell", Description: "SSLshell"},
	615:           {Sym: "sco-inetmgr", Description: "Internet Configuration Manager"},
	616:           {Sym: "sco-sysmgr", Description: "SCO System Administration Server"},
	617:           {Sym: "sco-dtmgr", Description: "SCO Desktop Administration Server"},
	618:           {Sym: "dei-icda", Description: "DEI-ICDA"},
	619:           {Sym: "compaq-evm", Description: "Compaq EVM"},
	620:           {Sym: "sco-websrvrmgr", Description: "SCO WebServer Manager"},
	621:           {Sym: "escp-ip", Description: "ESCP"},
	622:           {Sym: "collaborator", Description: "Collaborator"},
	623:           {Sym: "asf-rmcp", Description: "ASF Remote Management and Control Protocol"},
	624:           {Sym: "cryptoadmin", Description: "Crypto Admin"},
	625:           {Sym: "dec_dlm", Description: "DEC DLM"},
	626:           {Sym: "asia", Description: "ASIA"},
	627:           {Sym: "passgo-tivoli", Description: "PassGo Tivoli"},
	628:           {Sym: "qmqp", Description: "QMQP"},
	629:           {Sym: "3com-amp3", Description: "3Com AMP3"},
	630:           {Sym: "rda", Description: "RDA"},
	631:           {Sym: "ipp", Description: "IPP (Internet Printing Protocol)"},
	632:           {Sym: "bmpp", Description: "bmpp"},
	633:           {Sym: "servstat", Description: "Service Status update (Sterling Software)"},
	634:           {Sym: "ginad", Description: "ginad"},
	635:           {Sym: "rlzdbase", Description: "RLZ DBase"},
	636:           {Sym: "ldaps", Description: "ldap protocol over TLS/SSL (was sldap)"},
	637:           {Sym: "lanserver", Description: "lanserver"},
	638:           {Sym: "mcns-sec", Description: "mcns-sec"},
	639:           {Sym: "msdp", Description: "MSDP"},
	640:           {Sym: "entrust-sps", Description: "entrust-sps"},
	641:           {Sym: "repcmd", Description: "repcmd"},
	642:           {Sym: "esro-emsdp", Description: "ESRO-EMSDP V1.3"},
	643:           {Sym: "sanity", Description: "SANity"},
	644:           {Sym: "dwr", Description: "dwr"},
	645:           {Sym: "pssc", Description: "PSSC"},
	646:           {Sym: "ldp", Description: "LDP"},
	647:           {Sym: "dhcp-failover", Description: "DHCP Failover"},
	648:           {Sym: "rrp", Description: "Registry Registrar Protocol (RRP)"},
	649:           {Sym: "cadview-3d", Description: "Cadview-3d - streaming 3d models over the internet"},
	650:           {Sym: "obex", Description: "OBEX"},
	651:           {Sym: "ieee-mms", Description: "IEEE MMS"},
	652:           {Sym: "hello-port", Description: "HELLO_PORT"},
	653:           {Sym: "repscmd", Description: "RepCmd"},
	654:           {Sym: "aodv", Description: "AODV"},
	655:           {Sym: "tinc", Description: "TINC"},
	656:           {Sym: "spmp", Description: "SPMP"},
	657:           {Sym: "rmc", Description: "RMC"},
	658:           {Sym: "tenfold", Description: "TenFold"},
	660:           {Sym: "mac-srvr-admin", Description: "MacOS Server Admin"},
	661:           {Sym: "hap", Description: "HAP"},
	662:           {Sym: "pftp", Description: "PFTP"},
	663:           {Sym: "purenoise", Description: "PureNoise"},
	664:           {Sym: "asf-secure-rmcp", Description: "ASF Secure Remote Management and Control Protocol"},
	665:           {Sym: "sun-dr", Description: "Sun DR"},
	666:           {Sym: "mdqs"},
	667:           {Sym: "disclose", Description: "campaign contribution disclosures - SDR Technologies"},
	668:           {Sym: "mecomm", Description: "MeComm"},
	669:           {Sym: "meregister", Description: "MeRegister"},
	670:           {Sym: "vacdsm-sws", Description: "VACDSM-SWS"},
	671:           {Sym: "vacdsm-app", Description: "VACDSM-APP"},
	672:           {Sym: "vpps-qua", Description: "VPPS-QUA"},
	673:           {Sym: "cimplex", Description: "CIMPLEX"},
	674:           {Sym: "acap", Description: "ACAP"},
	675:           {Sym: "dctp", Description: "DCTP"},
	676:           {Sym: "vpps-via", Description: "VPPS Via"},
	677:           {Sym: "vpp", Description: "Virtual Presence Protocol"},
	678:           {Sym: "ggf-ncp", Description: "GNU Generation Foundation NCP"},
	679:           {Sym: "mrm", Description: "MRM"},
	680:           {Sym: "entrust-aaas", Description: "entrust-aaas"},
	681:           {Sym: "entrust-aams", Description: "entrust-aams"},
	682:           {Sym: "xfr", Description: "XFR"},
	683:           {Sym: "corba-iiop", Description: "CORBA IIOP"},
	684:           {Sym: "corba-iiop-ssl", Description: "CORBA IIOP SSL"},
	685:           {Sym: "mdc-portmapper", Description: "MDC Port Mapper"},
	686:           {Sym: "hcp-wismar", Description: "Hardware Control Protocol Wismar"},
	687:           {Sym: "asipregistry", Description: "asipregistry"},
	688:           {Sym: "realm-rusd", Description: "REALM-RUSD"},
	689:           {Sym: "nmap", Description: "NMAP"},
	690:           {Sym: "vatp", Description: "VATP"},
	691:           {Sym: "msexch-routing", Description: "MS Exchange Routing"},
	692:           {Sym: "hyperwave-isp", Description: "Hyperwave-ISP"},
	693:           {Sym: "connendp", Description: "connendp"},
	694:           {Sym: "ha-cluster", Description: "ha-cluster"},
	695:           {Sym: "ieee-mms-ssl", Description: "IEEE-MMS-SSL"},
	696:           {Sym: "rushd", Description: "RUSHD"},
	697:           {Sym: "uuidgen", Description: "UUIDGEN"},
	698:           {Sym: "olsr", Description: "OLSR"},
	699:           {Sym: "accessnetwork", Description: "Access Network"},
	700:           {Sym: "epp", Description: "Extensible Provisioning Protocol"},
	701:           {Sym: "lmp", Description: "Link Management Protocol (LMP)"},
	702:           {Sym: "iris-beep", Description: "IRIS over BEEP"},
	704:           {Sym: "elcsd", Description: "errlog copy/server daemon"},
	705:           {Sym: "agentx", Description: "AgentX"},
	706:           {Sym: "silc", Description: "SILC"},
	707:           {Sym: "borland-dsj", Description: "Borland DSJ"},
	709:           {Sym: "entrust-kmsh", Description: "Entrust Key Management Service Handler"},
	710:           {Sym: "entrust-ash", Description: "Entrust Administration Service Handler"},
	711:           {Sym: "cisco-tdp", Description: "Cisco TDP"},
	712:           {Sym: "tbrpf", Description: "TBRPF"},
	729:           {Sym: "netviewdm1", Description: "IBM NetView DM/6000 Server/Client"},
	730:           {Sym: "netviewdm2", Description: "IBM NetView DM/6000 send/tcp"},
	731:           {Sym: "netviewdm3", Description: "IBM NetView DM/6000 receive/tcp"},
	741:           {Sym: "netgw", Description: "netGW"},
	742:           {Sym: "netrcs", Description: "Network based Rev. Cont. Sys"},
	744:           {Sym: "flexlm", Description: "Flexible License Manager"},
	747:           {Sym: "fujitsu-dev", Description: "Fujitsu Device Control"},
	748:           {Sym: "ris-cm", Description: "Russell Info Sci Calendar Manager"},
	749:           {Sym: "kerberos-adm", Description: "kerberos administration"},
	750:           {Sym: "rfile"},
	751:           {Sym: "pump"},
	752:           {Sym: "qrh"},
	753:           {Sym: "rrh"},
	754:           {Sym: "tell", Description: "send"},
	758:           {Sym: "nlogin"},
	759:           {Sym: "con"},
	760:           {Sym: "ns"},
	761:           {Sym: "rxe"},
	762:           {Sym: "quotad"},
	763:           {Sym: "cycleserv"},
	764:           {Sym: "omserv"},
	765:           {Sym: "webster"},
	767:           {Sym: "phonebook", Description: "phone"},
	769:           {Sym: "vid"},
	770:           {Sym: "cadlock"},
	771:           {Sym: "rtip"},
	772:           {Sym: "cycleserv2"},
	773:           {Sym: "submit"},
	774:           {Sym: "rpasswd"},
	775:           {Sym: "entomb"},
	776:           {Sym: "wpages"},
	777:           {Sym: "multiling-http", Description: "Multiling HTTP"},
	780:           {Sym: "wpgs"},
	800:           {Sym: "mdbs_daemon"},
	801:           {Sym: "device"},
	810:           {Sym: "fcp-udp", Description: "FCP"},
	828:           {Sym: "itm-mcell-s", Description: "itm-mcell-s"},
	829:           {Sym: "pkix-3-ca-ra", Description: "PKIX-3 CA/RA"},
	830:           {Sym: "netconf-ssh", Description: "NETCONF over SSH"},
	831:           {Sym: "netconf-beep", Description: "NETCONF over BEEP"},
	832:           {Sym: "netconfsoaphttp", Description: "NETCONF for SOAP over HTTPS"},
	833:           {Sym: "netconfsoapbeep", Description: "NETCONF for SOAP over BEEP"},
	847:           {Sym: "dhcp-failover2", Description: "dhcp-failover 2"},
	848:           {Sym: "gdoi", Description: "GDOI"},
	860:           {Sym: "iscsi", Description: "iSCSI"},
	861:           {Sym: "owamp-control", Description: "OWAMP-Control"},
	873:           {Sym: "rsync", Description: "rsync"},
	886:           {Sym: "iclcnet-locate", Description: "ICL coNETion locate server"},
	887:           {Sym: "iclcnet_svinfo", Description: "ICL coNETion server info"},
	888:           {Sym: "accessbuilder", Description: "AccessBuilder"},
	900:           {Sym: "omginitialrefs", Description: "OMG Initial Refs"},
	901:           {Sym: "smpnameres", Description: "SMPNAMERES"},
	902:           {Sym: "ideafarm-chat", Description: "IDEAFARM-CHAT"},
	903:           {Sym: "ideafarm-catch", Description: "IDEAFARM-CATCH"},
	910:           {Sym: "kink", Description: "Kerberized Internet Negotiation of Keys (KINK)"},
	911:           {Sym: "xact-backup", Description: "xact-backup"},
	912:           {Sym: "apex-mesh", Description: "APEX relay-relay service"},
	913:           {Sym: "apex-edge", Description: "APEX endpoint-relay service"},
	989:           {Sym: "ftps-data", Description: "ftp protocol, data, over TLS/SSL"},
	990:           {Sym: "ftps", Description: "ftp protocol, control, over TLS/SSL"},
	991:           {Sym: "nas", Description: "Netnews Administration System"},
	992:           {Sym: "telnets", Description: "telnet protocol over TLS/SSL"},
	993:           {Sym: "imaps", Description: "imap4 protocol over TLS/SSL"},
	994:           {Sym: "ircs", Description: "irc protocol over TLS/SSL"},
	995:           {Sym: "pop3s", Description: "pop3 protocol over TLS/SSL (was spop3)"},
	996:           {Sym: "vsinet", Description: "vsinet"},
	997:           {Sym: "maitrd"},
	998:           {Sym: "busboy"},
	999:           {Sym: "garcon"},
	1000:          {Sym: "cadlock2"},
	1010:          {Sym: "surf", Description: "surf"},
	TCPPortRTMP:   {Sym: "rtmp", Description: "Real-Time Messaging Protocol"},
}
