/*
 *  $Id: multiprofile.c 22997 2020-12-07 12:37:56Z yeti-dn $
 *  Copyright (C) 2020 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/stats.h>
#include <libprocess/arithmetic.h>
#include <libprocess/gwyprocesstypes.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "preview.h"

#define MULTIPROF_RUN_MODES GWY_RUN_INTERACTIVE

enum {
    NARGS = 6,
    MAX_THICKNESS = 128,
};

typedef enum {
    MULTIPROF_MODE_PROFILES = 0,
    MULTIPROF_MODE_MEAN_RMS = 1,
    MULTIPROF_MODE_MIN_MAX  = 2,
    MULTIPROF_NMODES,
} MultiprofMode;

typedef struct {
    GwyAppDataId image[NARGS];
    gboolean enabled[NARGS];      /* NB: enables[0] is always TRUE */
    GwyAppDataId target_graph;
    GwyMaskingType masking;
    gboolean use_first_mask;
    gint thickness;
    MultiprofMode mode;
    guint display;                /* GUI only, not stored. */
    gdouble lineno_frac;
} MultiprofArgs;

typedef struct {
    MultiprofArgs *args;
    GwyContainer *mydata;
    GtkWidget *dialogue;
    GtkWidget *view;
    GtkWidget *graph;
    GtkWidget *masking;
    GtkWidget *use_first_mask;
    GtkWidget *image[NARGS];
    GtkWidget *enabled[NARGS];
    GtkWidget *display[NARGS];
    GtkObject *thickness;
    GtkObject *lineno;
    GwySelection *selection;
    GtkWidget *mode;
    GtkWidget *target_graph;

    gint yres;
    gboolean in_update;
} MultiprofControls;

static gboolean       module_register       (void);
static void           multiprofile          (GwyContainer *data,
                                             GwyRunType run);
static gboolean       dialogue              (MultiprofArgs *args,
                                             GwyDataField *dfield,
                                             GwyContainer *data,
                                             gint id);
static void           update_controls       (MultiprofControls *controls,
                                             MultiprofArgs *args);
static void           thickness_changed     (MultiprofControls *controls,
                                             GtkAdjustment *adj);
static void           lineno_changed        (MultiprofControls *controls,
                                             GtkAdjustment *adj);
static void           masking_changed       (GtkComboBox *combo,
                                             MultiprofControls *controls);
static void           use_first_mask_changed(MultiprofControls *controls,
                                             GtkToggleButton *button);
static void           mode_changed          (GtkComboBox *combo,
                                             MultiprofControls *controls);
static void           selection_changed     (MultiprofControls *controls,
                                             gint hint,
                                             GwySelection *selection);
static void           display_changed       (MultiprofControls *controls,
                                             GtkToggleButton *button);
static void           enabled_changed       (MultiprofControls *controls,
                                             GtkToggleButton *check);
static void           image_selected        (MultiprofControls *controls,
                                             GwyDataChooser *chooser);
static gboolean       image_filter          (GwyContainer *data,
                                             gint id,
                                             gpointer user_data);
static void           update_target_graphs  (MultiprofControls *controls);
static void           target_graph_changed  (MultiprofControls *controls,
                                             GwyDataChooser *chooser);
static void           update_sensitivity    (MultiprofControls *controls);
static void           preview               (MultiprofControls *controls);
static GwyGraphModel* multiprofile_do       (const MultiprofArgs *args);
static void           load_args             (GwyContainer *container,
                                             MultiprofArgs *args);
static void           save_args             (GwyContainer *container,
                                             MultiprofArgs *args);
static void           sanitize_args         (MultiprofArgs *args);

static const MultiprofArgs defaults = {
    {
        GWY_APP_DATA_ID_NONE,   /* Always set to the current image. */
        GWY_APP_DATA_ID_NONE,
        GWY_APP_DATA_ID_NONE,
        GWY_APP_DATA_ID_NONE,
        GWY_APP_DATA_ID_NONE,
    },
    { TRUE, TRUE, FALSE, FALSE, FALSE },
    GWY_APP_DATA_ID_NONE,
    GWY_MASK_IGNORE, TRUE,
    1,
    MULTIPROF_MODE_PROFILES,
    0,
    0.5,
};

static GwyAppDataId image_ids[NARGS];
static GwyAppDataId target_id = GWY_APP_DATA_ID_NONE;

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Displays and extracts scan line graphs from multiple images "
       "simultaneously."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2020",
};

GWY_MODULE_QUERY2(module_info, multiprofile)

static gboolean
module_register(void)
{
    guint i;

    for (i = 0; i < NARGS; i++) {
        image_ids[i].datano = 0;
        image_ids[i].id = -1;
    }

    gwy_process_func_register("multiprofile",
                              (GwyProcessFunc)&multiprofile,
                              N_("/M_ultidata/_Multiprofile..."),
                              GWY_STOCK_PROFILE_MULTIPLE,
                              MULTIPROF_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Read lines from multiple images "
                                 "simultaneously"));

    return TRUE;
}

static void
multiprofile(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield;
    GwyGraphModel *gmodel;
    MultiprofArgs args;
    gint id;
    gboolean ok;

    g_return_if_fail(run & MULTIPROF_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield);
    load_args(gwy_app_settings_get(), &args);

    args.image[0].datano = gwy_app_data_browser_get_number(data);
    args.image[0].id = id;
    args.enabled[0] = TRUE;
    args.display = 0;

    if (run == GWY_RUN_INTERACTIVE) {
        ok = dialogue(&args, dfield, data, id);
        save_args(gwy_app_settings_get(), &args);
        if (!ok)
            return;
    }

    gmodel = multiprofile_do(&args);
    gwy_app_add_graph_or_curves(gmodel, data, &args.target_graph, 1);
    g_object_unref(gmodel);
}

static inline gdouble
frac_to_rowno(gdouble frac, gint res)
{
    return CLAMP(floor(frac*res), 0, res-1);
}

static inline gdouble
rowno_to_frac(gint lineno, gint res)
{
    return res > 1 ? lineno/(res - 1.0) : 0.5;
}

static gboolean
dialogue(MultiprofArgs *args,
         GwyDataField *dfield, GwyContainer *data, gint id)
{
    GtkWidget *dialogue, *table, *label, *hbox, *check, *button;
    GwyDataChooser *chooser;
    GwyGraphModel *gmodel;
    MultiprofControls controls;
    GSList *group = NULL;
    gint row, response;
    gchar *s;
    guint i;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.in_update = TRUE;
    controls.yres = gwy_data_field_get_yres(dfield);

    dialogue = gtk_dialog_new_with_buttons(_("Multiprofile"), NULL, 0,
                                           NULL);
    gtk_dialog_add_button(GTK_DIALOG(dialogue), _("_Reset"), RESPONSE_RESET);
    gtk_dialog_add_button(GTK_DIALOG(dialogue),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialogue),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);
    controls.dialogue = dialogue;

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       TRUE, TRUE, 4);

    controls.mydata = gwy_container_new();
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SMALL_SIZE,
                                   FALSE);
    controls.selection = create_vector_layer(GWY_DATA_VIEW(controls.view), 0,
                                             "Axis", TRUE);
    g_object_set(controls.selection,
                 "orientation", GWY_ORIENTATION_HORIZONTAL,
                 "max-objects", 1,
                 NULL);
    g_signal_connect_swapped(controls.selection, "changed",
                             G_CALLBACK(selection_changed), &controls);

    gtk_box_pack_start(GTK_BOX(hbox), controls.view, FALSE, FALSE, 0);

    gmodel = gwy_graph_model_new();
    controls.graph = gwy_graph_new(gmodel);
    gwy_graph_model_set_units_from_data_field(gmodel, dfield, 1, 0, 0, 1);
    gtk_widget_set_size_request(controls.graph,
                                PREVIEW_SIZE, PREVIEW_SMALL_SIZE);
    gwy_graph_enable_user_input(GWY_GRAPH(controls.graph), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), controls.graph, TRUE, TRUE, 0);

    hbox = gtk_hbox_new(FALSE, 20);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       FALSE, FALSE, 4);

    table = gtk_table_new(1+NARGS, 4, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, FALSE, FALSE, 4);
    row = 0;

    label = gwy_label_new_header(_("Images"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 3, row, row+1, GTK_FILL, 0, 0, 0);

    label = gtk_label_new(_("Show"));
    gtk_table_attach(GTK_TABLE(table), label,
                     3, 4, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    for (i = 0; i < NARGS; i++) {
        s = g_strdup_printf("%u", i+1);
        label = gtk_label_new(s);
        g_free(s);
        gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
        gtk_table_attach(GTK_TABLE(table), label,
                         0, 1, row, row+1, GTK_FILL, 0, 0, 0);

        controls.enabled[i] = check = gtk_check_button_new();
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check),
                                     args->enabled[i]);
        g_object_set_data(G_OBJECT(check), "id", GUINT_TO_POINTER(i));
        /* Not showing any check box looks odd, but it needs to be always
         * checked. */
        if (!i)
            gtk_widget_set_sensitive(check, FALSE);
        gtk_table_attach(GTK_TABLE(table), check,
                         1, 2, row, row+1, GTK_FILL, 0, 0, 0);

        controls.image[i] = gwy_data_chooser_new_channels();
        g_object_set_data(G_OBJECT(controls.image[i]),
                          "id", GUINT_TO_POINTER(i));
        gtk_table_attach(GTK_TABLE(table), controls.image[i],
                         2, 3, row, row+1, GTK_FILL, 0, 0, 0);

        controls.display[i] = button = gtk_radio_button_new(group);
        g_object_set_data(G_OBJECT(button), "id", GUINT_TO_POINTER(i));
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), !i);
        group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
        gtk_table_attach(GTK_TABLE(table), button,
                         3, 4, row, row+1, GTK_FILL, 0, 0, 0);
        row++;
    }

    chooser = GWY_DATA_CHOOSER(controls.image[0]);
    gwy_data_chooser_set_active_id(chooser, args->image + 0);

    for (i = 1; i < NARGS; i++) {
        chooser = GWY_DATA_CHOOSER(controls.image[i]);
        gwy_data_chooser_set_filter(chooser, image_filter, args, NULL);
        gwy_data_chooser_set_active_id(chooser, args->image + i);
        gwy_data_chooser_get_active_id(chooser, args->image + i);
    }

    for (i = 0; i < NARGS; i++) {
        g_signal_connect_swapped(controls.enabled[i], "toggled",
                                 G_CALLBACK(enabled_changed), &controls);
        g_signal_connect_swapped(controls.image[i], "changed",
                                 G_CALLBACK(image_selected), &controls);
        g_signal_connect_swapped(controls.display[i], "toggled",
                                 G_CALLBACK(display_changed), &controls);
    }

    table = gtk_table_new(7, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, FALSE, FALSE, 4);
    row = 0;

    label = gwy_label_new_header(_("Options"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.lineno = gtk_adjustment_new(frac_to_rowno(args->lineno_frac,
                                                       controls.yres),
                                         0, controls.yres-1, 1, 10, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row++, _("_Scan line:"), NULL,
                            controls.lineno,
                            GWY_HSCALE_LINEAR | GWY_HSCALE_SNAP);
    g_signal_connect_swapped(controls.lineno, "value-changed",
                             G_CALLBACK(lineno_changed), &controls);

    controls.thickness = gtk_adjustment_new(args->thickness,
                                            1, MAX_THICKNESS, 1, 10, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row++, _("_Thickness:"), _("px"),
                            controls.thickness,
                            GWY_HSCALE_SQRT | GWY_HSCALE_SNAP);
    g_signal_connect_swapped(controls.thickness, "value-changed",
                             G_CALLBACK(thickness_changed), &controls);

    controls.masking = gwy_enum_combo_box_new(gwy_masking_type_get_enum(), -1,
                                              G_CALLBACK(masking_changed),
                                              &controls,
                                              args->masking, TRUE);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row++, _("_Masking:"), NULL,
                            GTK_OBJECT(controls.masking),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.use_first_mask
        = gtk_check_button_new_with_mnemonic(_("Use _first mask "
                                               "for all images"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.use_first_mask),
                                 args->use_first_mask);
    gtk_table_attach(GTK_TABLE(table), controls.use_first_mask,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls.use_first_mask, "toggled",
                             G_CALLBACK(use_first_mask_changed), &controls);
    row++;

    controls.mode
        = gwy_enum_combo_box_newl(G_CALLBACK(mode_changed), &controls,
                                  args->mode,
                                  _("All profiles"),
                                  MULTIPROF_MODE_PROFILES,
                                  _("Mean and deviation"),
                                  MULTIPROF_MODE_MEAN_RMS,
                                  _("Minimum and maximum"),
                                  MULTIPROF_MODE_MIN_MAX,
                                  NULL);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row++, _("_Mode:"), NULL,
                            GTK_OBJECT(controls.mode),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    controls.target_graph = create_target_graph(&args->target_graph,
                                                GTK_WIDGET(table), row++,
                                                gmodel);
    g_object_unref(gmodel);
    g_signal_connect_swapped(controls.target_graph, "changed",
                             G_CALLBACK(target_graph_changed), &controls);

    update_sensitivity(&controls);
    controls.in_update = FALSE;
    lineno_changed(&controls, GTK_ADJUSTMENT(controls.lineno));

    gtk_widget_show_all(dialogue);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialogue));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialogue);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            controls.in_update = TRUE;
            gwy_assign(args->enabled, defaults.enabled, NARGS);
            args->masking = defaults.masking;
            args->use_first_mask = defaults.use_first_mask;
            args->thickness = defaults.thickness;
            args->mode = defaults.mode;
            args->display = defaults.display;
            args->lineno_frac = defaults.lineno_frac;
            update_controls(&controls, args);
            controls.in_update = FALSE;
            preview(&controls);
            break;

            case RESPONSE_PREVIEW:
            preview(&controls);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialogue);

    return TRUE;
}

static void
update_controls(MultiprofControls *controls, MultiprofArgs *args)
{
    guint lineno, i;

    i = args->display;
    lineno = frac_to_rowno(args->lineno_frac, controls->yres);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->mode), args->mode);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->lineno), lineno);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->thickness),
                             args->thickness);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->use_first_mask),
                                 args->use_first_mask);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->display[i]), TRUE);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->masking),
                                  args->masking);

    for (i = 1; i < NARGS; i++) {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->enabled[i]),
                                     args->enabled[i]);
    }
}

static GwyDataField*
get_chosen_image(const MultiprofArgs *args, guint i, gboolean want_mask)
{
    const GwyAppDataId *dataid = args->image + i;
    GwyContainer *data;
    GwyDataField *field = NULL;
    GQuark quark;

    if (!args->enabled[i])
        return NULL;
    if (dataid->datano <= 0 || dataid->id < 0)
        return NULL;

    data = gwy_app_data_browser_get(args->image[i].datano);
    if (want_mask)
        quark = gwy_app_get_mask_key_for_id(args->image[i].id);
    else
        quark = gwy_app_get_data_key_for_id(args->image[i].id);

    gwy_container_gis_object(data, quark, (GObject**)&field);
    return field;
}

static void
thickness_changed(MultiprofControls *controls, GtkAdjustment *adj)
{
    controls->args->thickness = gwy_adjustment_get_int(adj);
    preview(controls);
}

static void
lineno_changed(MultiprofControls *controls, GtkAdjustment *adj)
{
    MultiprofArgs *args = controls->args;
    GwyDataField *dfield;
    gint lineno;
    gdouble y;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->lineno_frac = rowno_to_frac(gwy_adjustment_get_int(adj),
                                      controls->yres);
    dfield = get_chosen_image(args, 0, FALSE);
    lineno = frac_to_rowno(args->lineno_frac, controls->yres);
    y = gwy_data_field_itor(dfield, lineno + 0.5);
    gwy_selection_set_data(controls->selection, 1, &y);
    controls->in_update = FALSE;
    preview(controls);
}

static void
selection_changed(MultiprofControls *controls,
                  G_GNUC_UNUSED gint hint,
                  GwySelection *selection)
{
    MultiprofArgs *args = controls->args;
    GwyDataField *dfield;
    gint lineno;
    gdouble y;

    if (controls->in_update)
        return;

    if (!gwy_selection_get_object(selection, 0, &y))
        return;

    controls->in_update = TRUE;
    dfield = get_chosen_image(args, 0, FALSE);
    lineno = CLAMP(GWY_ROUND(gwy_data_field_rtoi(dfield, y)),
                   0, controls->yres-1);
    args->lineno_frac = rowno_to_frac(lineno, controls->yres);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->lineno), lineno);
    controls->in_update = FALSE;
    preview(controls);
}

static void
masking_changed(GtkComboBox *combo, MultiprofControls *controls)
{
    controls->args->masking = gwy_enum_combo_box_get_active(combo);
    update_sensitivity(controls);
    preview(controls);
}

static void
use_first_mask_changed(MultiprofControls *controls, GtkToggleButton *button)
{
    controls->args->use_first_mask = gtk_toggle_button_get_active(button);
    preview(controls);
}

static void
mode_changed(GtkComboBox *combo, MultiprofControls *controls)
{
    controls->args->mode = gwy_enum_combo_box_get_active(combo);
    preview(controls);
}

static void
display_changed(MultiprofControls *controls, GtkToggleButton *button)
{
    MultiprofArgs *args = controls->args;
    GwyDataField *dfield;
    guint i;

    if (!gtk_toggle_button_get_active(button))
        return;

    i = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(button), "id"));
    args->display = i;

    dfield = get_chosen_image(args, i, FALSE);
    g_return_if_fail(GWY_IS_DATA_FIELD(dfield));
    gwy_container_set_object_by_name(controls->mydata, "/0/data", dfield);
}

static void
enabled_changed(MultiprofControls *controls, GtkToggleButton *check)
{
    MultiprofArgs *args = controls->args;
    guint i;

    i = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(check), "id"));
    args->enabled[i] = gtk_toggle_button_get_active(check);

    /* When we disable an image also stop showing it. */
    if (!args->enabled[i]) {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->display[0]),
                                     TRUE);
    }
    update_sensitivity(controls);
    preview(controls);
}

static void
image_selected(MultiprofControls *controls, GwyDataChooser *chooser)
{
    MultiprofArgs *args = controls->args;
    GwyDataField *dfield;
    guint i, j;
    gint lineno;
    gdouble y;

    i = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(chooser), "id"));
    gwy_data_chooser_get_active_id(chooser, args->image + i);
    if (controls->in_update)
        return;

    if (!i) {
        controls->in_update = TRUE;
        for (j = 1; j < NARGS; j++) {
            chooser = GWY_DATA_CHOOSER(controls->image[j]);
            gwy_data_chooser_refilter(chooser);
        }

        dfield = get_chosen_image(args, 0, FALSE);
        controls->yres = gwy_data_field_get_yres(dfield);
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->lineno),
                                 frac_to_rowno(args->lineno_frac,
                                               controls->yres));

        lineno = frac_to_rowno(args->lineno_frac, controls->yres);
        y = gwy_data_field_itor(dfield, lineno + 0.5);
        gwy_selection_set_data(controls->selection, 1, &y);

        controls->in_update = FALSE;
    }

    preview(controls);
}

static gboolean
image_filter(GwyContainer *otherdata,
             gint otherid,
             gpointer user_data)
{
    MultiprofArgs *args = (MultiprofArgs*)user_data;
    GwyContainer *firstdata;
    GwyDataField *firstfield, *otherfield;
    GQuark quark;

    if (!otherdata || otherid < 0)
        return FALSE;

    quark = gwy_app_get_data_key_for_id(args->image[0].id);
    firstdata = gwy_app_data_browser_get(args->image[0].datano);
    /* Should not happen. */
    if (!gwy_container_gis_object(firstdata, quark, (GObject**)&firstfield))
        return FALSE;

    quark = gwy_app_get_data_key_for_id(otherid);
    if (!gwy_container_gis_object(otherdata, quark, (GObject**)&otherfield))
        return FALSE;

    return !gwy_data_field_check_compatibility(otherfield, firstfield,
                                               GWY_DATA_COMPATIBILITY_ALL);
}

static void
update_target_graphs(MultiprofControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);
    gwy_data_chooser_refilter(chooser);
}

static void
target_graph_changed(MultiprofControls *controls, GwyDataChooser *chooser)
{
    gwy_data_chooser_get_active_id(chooser, &controls->args->target_graph);
}

static void
update_sensitivity(MultiprofControls *controls)
{
    const MultiprofArgs *args = controls->args;
    guint i;

    for (i = 0; i < NARGS; i++) {
        gtk_widget_set_sensitive(controls->image[i], args->enabled[i]);
        gtk_widget_set_sensitive(controls->display[i], args->enabled[i]);
    }
    gtk_widget_set_sensitive(controls->use_first_mask,
                             args->masking != GWY_MASK_IGNORE);
}

static void
preview(MultiprofControls *controls)
{
    MultiprofArgs *args = controls->args;
    GwyGraphModel *gmodel;

    if (controls->in_update)
        return;

    gmodel = multiprofile_do(args);
    set_graph_model_keeping_label_pos(GWY_GRAPH(controls->graph), gmodel);
    update_target_graphs(controls);
}

/* XXX: Duplicate code with tools/cprofile.c */
static void
extract_row_profile(GwyDataField *dfield,
                    GwyDataField *mask, GwyMaskingType masking,
                    GArray *xydata, gint row, gint thickness)
{
    gint xres, yres, ifrom, ito, i, j;
    const gdouble *d, *drow, *m, *mrow;
    gdouble dx;
    GwyXY *xy;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    dx = gwy_data_field_get_dx(dfield);
    d = gwy_data_field_get_data_const(dfield);

    g_array_set_size(xydata, xres);
    xy = &g_array_index(xydata, GwyXY, 0);
    gwy_clear(xy, xres);

    ifrom = row - (thickness - 1)/2;
    ifrom = MAX(ifrom, 0);
    ito = row + thickness/2 + 1;
    ito = MIN(ito, yres);

    if (mask && masking != GWY_MASK_IGNORE)
        m = gwy_data_field_get_data_const(mask);
    else {
        m = NULL;
        for (j = 0; j < xres; j++)
            xy[j].x = ito - ifrom;
    }

    for (i = ifrom; i < ito; i++) {
        drow = d + i*xres;
        if (m) {
            mrow = m + i*xres;
            if (masking == GWY_MASK_INCLUDE) {
                for (j = 0; j < xres; j++) {
                    if (mrow[j] > 0.0) {
                        xy[j].y += drow[j];
                        xy[j].x += 1.0;
                    }
                }
            }
            else {
                for (j = 0; j < xres; j++) {
                    if (mrow[j] <= 0.0) {
                        xy[j].y += drow[j];
                        xy[j].x += 1.0;
                    }
                }
            }
        }
        else {
            for (j = 0; j < xres; j++)
                xy[j].y += drow[j];
        }
    }

    for (i = j = 0; j < xres; j++) {
        if (xy[j].x > 0.0) {
            xy[i].y = xy[j].y/xy[j].x;
            xy[i].x = dx*j;
            i++;
        }
    }
    g_array_set_size(xydata, i);
}

static GwyGraphModel*
multiprofile_do_profiles(const MultiprofArgs *args)
{
    GwyGraphModel *gmodel;
    GwyGraphCurveModel *gcmodel;
    GwyDataField *dfield, *mfield = NULL;
    GArray *xydata;
    gint lineno, yres;
    gchar *title;
    guint i;

    dfield = get_chosen_image(args, 0, FALSE);
    g_return_val_if_fail(GWY_IS_DATA_FIELD(dfield), NULL);
    yres = gwy_data_field_get_yres(dfield);
    lineno = frac_to_rowno(args->lineno_frac, yres);

    gmodel = gwy_graph_model_new();
    gwy_graph_model_set_units_from_data_field(gmodel, dfield, 1, 0, 0, 1);

    xydata = g_array_new(FALSE, FALSE, sizeof(GwyXY));
    for (i = 0; i < NARGS; i++) {
        if (!(dfield = get_chosen_image(args, i, FALSE)))
            continue;

        if (args->masking != GWY_MASK_IGNORE)
            mfield = get_chosen_image(args, args->use_first_mask ? 0 : i, TRUE);

        extract_row_profile(dfield, mfield, args->masking, xydata,
                            lineno, args->thickness);
        gcmodel = gwy_graph_curve_model_new();
        gwy_graph_curve_model_set_data_interleaved(gcmodel,
                                                   (gdouble*)xydata->data,
                                                   xydata->len);
        title = g_strdup_printf("%u", i+1);
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "color", gwy_graph_get_preset_color(i),
                     "description", title,
                     NULL);
        g_free(title);

        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
    }
    g_array_free(xydata, TRUE);

    return gmodel;
}

static GwyGraphCurveModel*
add_curve(GwyGraphModel *gmodel, GwyDataLine *dline, GwyDataLine *weight,
          const GwyRGBA *colour, const gchar *description,
          GArray *xydata)
{
    GwyGraphCurveModel *gcmodel;
    const gdouble *w, *d;
    guint i, res;
    gdouble dx;
    GwyXY xy;

    res = gwy_data_line_get_res(dline);
    dx = gwy_data_line_get_dx(dline);
    d = gwy_data_line_get_data(dline);
    w = gwy_data_line_get_data(weight);

    g_array_set_size(xydata, 0);
    for (i = 0; i < res; i++) {
        if (!w[i])
            continue;

        xy.x = dx*i;
        xy.y = d[i];
        g_array_append_val(xydata, xy);
    }
    gcmodel = gwy_graph_curve_model_new();
    gwy_graph_curve_model_set_data_interleaved(gcmodel,
                                               (gdouble*)xydata->data,
                                               xydata->len);
    g_object_set(gcmodel,
                 "mode", GWY_GRAPH_CURVE_LINE,
                 "color", colour,
                 "description", description,
                 NULL);
    gwy_graph_model_add_curve(gmodel, gcmodel);
    g_object_unref(gcmodel);

    return gcmodel;
}


static GwyGraphModel*
multiprofile_do_stats(const MultiprofArgs *args)
{
    static const GwyRGBA upper_colour = { 1.000, 0.386, 0.380, 1.000 };
    static const GwyRGBA lower_colour = { 0.380, 0.625, 1.000, 1.000 };

    GwyGraphModel *gmodel;
    GwyDataField *dfield, *mfield, *bigfield, *bigmask = NULL;
    GwyDataLine *avg, *aux1, *aux2, *weight;
    GArray *xydata;
    gint lineno, xres, yres, ifrom, ito, blockheight;
    gdouble dx;
    guint i, nimages;

    dfield = get_chosen_image(args, 0, FALSE);
    g_return_val_if_fail(GWY_IS_DATA_FIELD(dfield), NULL);
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    dx = gwy_data_field_get_dx(dfield);
    lineno = frac_to_rowno(args->lineno_frac, yres);

    ifrom = lineno - (args->thickness - 1)/2;
    ifrom = MAX(ifrom, 0);
    ito = lineno + args->thickness/2 + 1;
    ito = MIN(ito, yres);
    blockheight = ito - ifrom;

    gmodel = gwy_graph_model_new();
    gwy_graph_model_set_units_from_data_field(gmodel, dfield, 1, 0, 0, 1);

    nimages = 0;
    for (i = 0; i < NARGS; i++) {
        if (get_chosen_image(args, i, FALSE))
            nimages++;
    }
    g_return_val_if_fail(nimages, NULL);

    bigfield = gwy_data_field_new(xres, nimages*blockheight, dx*xres, 1.0,
                                  FALSE);
    if (args->masking != GWY_MASK_IGNORE)
        bigmask = gwy_data_field_new_alike(bigfield, FALSE);

    nimages = 0;
    for (i = 0; i < NARGS; i++) {
        if (!(dfield = get_chosen_image(args, i, FALSE)))
            continue;

        gwy_data_field_area_copy(dfield, bigfield,
                                 0, ifrom, xres, blockheight,
                                 0, nimages*blockheight);

        if (bigmask) {
            mfield = get_chosen_image(args, i, TRUE);
            if (mfield) {
                gwy_data_field_area_copy(mfield, bigmask,
                                         0, ifrom, xres, blockheight,
                                         0, nimages*blockheight);
            }
            else {
                gwy_data_field_area_fill(bigmask,
                                         0, nimages*blockheight,
                                         xres, blockheight,
                                         args->masking == GWY_MASK_INCLUDE);
            }
        }
        nimages++;
    }

    xydata = g_array_new(FALSE, FALSE, sizeof(GwyXY));
    avg = gwy_data_line_new(1, 1.0, FALSE);
    weight = gwy_data_line_new(1, 1.0, FALSE);

    gwy_data_field_get_line_stats_mask(bigfield,
                                       bigmask, args->masking,
                                       avg, weight,
                                       0, 0, xres, nimages*blockheight,
                                       GWY_LINE_STAT_MEAN,
                                       GWY_ORIENTATION_VERTICAL);
    add_curve(gmodel, avg, weight,
              gwy_graph_get_preset_color(0), _("Mean"), xydata);

    aux1 = gwy_data_line_new_alike(avg, FALSE);
    aux2 = gwy_data_line_new_alike(avg, FALSE);
    if (args->mode == MULTIPROF_MODE_MEAN_RMS) {
        gwy_data_field_get_line_stats_mask(bigfield,
                                           bigmask, args->masking,
                                           aux1, NULL,
                                           0, 0, xres, nimages*blockheight,
                                           GWY_LINE_STAT_RMS,
                                           GWY_ORIENTATION_VERTICAL);
        gwy_data_line_subtract_lines(aux2, avg, aux1);
        add_curve(gmodel, aux2, weight, &lower_colour, _("Lower"), xydata);
        gwy_data_line_sum_lines(aux2, avg, aux1);
        add_curve(gmodel, aux2, weight, &upper_colour, _("Upper"), xydata);
    }
    else {
        gwy_data_field_get_line_stats_mask(bigfield,
                                           bigmask, args->masking,
                                           aux1, NULL,
                                           0, 0, xres, nimages*blockheight,
                                           GWY_LINE_STAT_MINIMUM,
                                           GWY_ORIENTATION_VERTICAL);
        gwy_data_field_get_line_stats_mask(bigfield,
                                           bigmask, args->masking,
                                           aux2, NULL,
                                           0, 0, xres, nimages*blockheight,
                                           GWY_LINE_STAT_MAXIMUM,
                                           GWY_ORIENTATION_VERTICAL);
        add_curve(gmodel, aux1, weight, &lower_colour, _("Lower"), xydata);
        add_curve(gmodel, aux2, weight, &upper_colour, _("Upper"), xydata);
    }

    g_object_unref(avg);
    g_object_unref(aux1);
    g_object_unref(aux2);
    g_object_unref(weight);
    g_object_unref(bigfield);
    GWY_OBJECT_UNREF(bigmask);
    g_array_free(xydata, TRUE);

    return gmodel;
}

static GwyGraphModel*
multiprofile_do(const MultiprofArgs *args)
{
    if (args->mode == MULTIPROF_MODE_PROFILES)
        return multiprofile_do_profiles(args);

    return multiprofile_do_stats(args);
}

static const gchar enabled_prefix[]     = "/module/multiprofile/enabled";
static const gchar lineno_frac_key[]    = "/module/multiprofile/lineno_frac";
static const gchar masking_key[]        = "/module/multiprofile/masking";
static const gchar mode_key[]           = "/module/multiprofile/mode";
static const gchar thickness_key[]      = "/module/multiprofile/thickness";
static const gchar use_first_mask_key[] = "/module/multiprofile/use_first_mask";

static void
sanitize_args(MultiprofArgs *args)
{
    guint i;

    args->masking = gwy_enum_sanitize_value(args->masking,
                                            GWY_TYPE_MASKING_TYPE);
    args->mode = CLAMP(args->mode, 0, MULTIPROF_NMODES-1);
    args->thickness = CLAMP(args->thickness, 1, MAX_THICKNESS);
    args->lineno_frac = CLAMP(args->lineno_frac, 0.0, 1.0);
    args->use_first_mask = !!args->use_first_mask;
    for (i = 0; i < NARGS; i++) {
        gwy_app_data_id_verify_channel(args->image + i);
        args->enabled[i] = !!args->enabled[i];

    }
    gwy_app_data_id_verify_graph(&args->target_graph);
}

static void
load_args(GwyContainer *container,
          MultiprofArgs *args)
{
    gchar buf[40];
    guint i;

    *args = defaults;
    gwy_container_gis_enum_by_name(container, masking_key, &args->masking);
    gwy_container_gis_enum_by_name(container, mode_key, &args->mode);
    gwy_container_gis_int32_by_name(container, thickness_key, &args->thickness);
    gwy_container_gis_double_by_name(container, lineno_frac_key,
                                     &args->lineno_frac);
    gwy_container_gis_boolean_by_name(container, use_first_mask_key,
                                      &args->use_first_mask);
    gwy_assign(args->image, image_ids, NARGS);
    for (i = 0; i < NARGS; i++) {
        g_snprintf(buf, sizeof(buf), "%s/%d", enabled_prefix, i);
        gwy_container_gis_boolean_by_name(container, buf, args->enabled + i);

    }
    args->target_graph = target_id;
    sanitize_args(args);
}

static void
save_args(GwyContainer *container,
          MultiprofArgs *args)
{
    gchar buf[40];
    guint i;

    gwy_assign(image_ids, args->image, NARGS);
    target_id = args->target_graph;
    for (i = 0; i < NARGS; i++) {
        g_snprintf(buf, sizeof(buf), "%s/%d", enabled_prefix, i);
        gwy_container_set_boolean_by_name(container, buf, args->enabled[i]);

    }
    gwy_container_set_enum_by_name(container, masking_key, args->masking);
    gwy_container_set_enum_by_name(container, mode_key, args->mode);
    gwy_container_set_int32_by_name(container, thickness_key, args->thickness);
    gwy_container_set_double_by_name(container, lineno_frac_key,
                                     args->lineno_frac);
    gwy_container_set_boolean_by_name(container, use_first_mask_key,
                                      args->use_first_mask);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
