/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#include "stdlib/strided/common/binary.h"
#include <node_api.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdbool.h>
#include <assert.h>

/**
* Adds two doubles.
*
* @private
* @param x  first double
* @param y  second double
* @return   sum
*/
static inline double add( double x, double y ) {
	return x + y;
}

/**
* Add-on namespace.
*/
namespace addon_strided_add {

	/**
	* Adds each element in `X` to a corresponding element in `Y` and assigns the result to an element in `Z`.
	*
	* ## Notes
	*
	* -   When called from JavaScript, the function expects the following arguments:
	*
	*     -   `N`: number of indexed elements
	*     -   `X`: input array
	*     -   `strideX`: `X` stride length
	*     -   `Y`: input array
	*     -   `strideY`: `Y` stride length
	*     -   `Z`: destination array
	*     -   `strideZ`: `Z` stride length
	*/
	 napi_value node_add( napi_env env, napi_callback_info info ) {
		napi_status status;
		uint8_t *arrays[ 3 ];
		int64_t strides[ 3 ];
		int64_t shape[ 1 ];

		size_t argc = 7;
		napi_value argv[ 7 ];
		status = napi_get_cb_info( env, info, &argc, argv, nullptr, nullptr );
		assert( status == napi_ok );

		if ( argc < 7 ) {
			napi_throw_error( env, nullptr, "invalid invocation. Must provide 7 arguments." );
			return nullptr;
		}

		napi_valuetype vtype0;
		status = napi_typeof( env, argv[ 0 ], &vtype0 );
		assert( status == napi_ok );
		if ( vtype0 != napi_number ) {
			napi_throw_type_error( env, nullptr, "invalid argument. First argument must be a number." );
			return nullptr;
		}

		bool res1;
		status = napi_is_typedarray( env, argv[ 1 ], &res1 );
		assert( status == napi_ok );
		if ( res1 == false ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Second argument must be a Float64Array." );
			return nullptr;
		}

		napi_valuetype vtype2;
		status = napi_typeof( env, argv[ 2 ], &vtype2 );
		assert( status == napi_ok );
		if ( vtype2 != napi_number ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Third argument must be a number." );
			return nullptr;
		}

		bool res3;
		status = napi_is_typedarray( env, argv[ 3 ], &res3 );
		assert( status == napi_ok );
		if ( res3 == false ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Fourth argument must be a Float64Array." );
			return nullptr;
		}

		napi_valuetype vtype4;
		status = napi_typeof( env, argv[ 4 ], &vtype4 );
		assert( status == napi_ok );
		if ( vtype4 != napi_number ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Fifth argument must be a number." );
			return nullptr;
		}

		bool res5;
		status = napi_is_typedarray( env, argv[ 5 ], &res5 );
		assert( status == napi_ok );
		if ( res5 == false ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Sixth argument must be a Float64Array." );
			return nullptr;
		}

		napi_valuetype vtype6;
		status = napi_typeof( env, argv[ 6 ], &vtype6 );
		assert( status == napi_ok );
		if ( vtype6 != napi_number ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Seventh argument must be a number." );
			return nullptr;
		}

		// Retrieve the number of elements:
		int64_t N;
		status = napi_get_value_int64( env, argv[ 0 ], &N );
		assert( status == napi_ok );

		shape[ 0 ] = N;

		// Retrieve the strides:
		int64_t strideX;
		status = napi_get_value_int64( env, argv[ 2 ], &strideX );
		assert( status == napi_ok );

		int64_t strideY;
		status = napi_get_value_int64( env, argv[ 4 ], &strideY );
		assert( status == napi_ok );

		int64_t strideZ;
		status = napi_get_value_int64( env, argv[ 6 ], &strideZ );
		assert( status == napi_ok );

		// Convert the strides from units of elements to units of bytes:
		strides[ 0 ] = strideX * 8; // 8 bytes per double
		strides[ 1 ] = strideY * 8;
		strides[ 2 ] = strideZ * 8;

		// Get the underlying byte arrays:
		napi_typedarray_type vtype1;
		size_t xlen;
		void *X;
		status = napi_get_typedarray_info( env, argv[ 1 ], &vtype1, &xlen, &X, nullptr, nullptr );
		assert( status == napi_ok );
		if ( vtype1 != napi_float64_array ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Second argument must be a Float64Array." );
			return nullptr;
		}
		if ( (N-1)*llabs(strideX) >= (int64_t)xlen ) {
			napi_throw_range_error( env, nullptr, "invalid argument. Second argument has insufficient elements based on the associated stride and the number of indexed elements." );
			return nullptr;
		}

		napi_typedarray_type vtype3;
		size_t ylen;
		void *Y;
		status = napi_get_typedarray_info( env, argv[ 3 ], &vtype3, &ylen, &Y, nullptr, nullptr );
		assert( status == napi_ok );
		if ( vtype3 != napi_float64_array ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Fourth argument must be a Float64Array." );
			return nullptr;
		}
		if ( (N-1)*llabs(strideY) >= (int64_t)ylen ) {
			napi_throw_range_error( env, nullptr, "invalid argument. Fourth argument has insufficient elements based on the associated stride and the number of indexed elements." );
			return nullptr;
		}

		napi_typedarray_type vtype5;
		size_t zlen;
		void *Z;
		status = napi_get_typedarray_info( env, argv[ 5 ], &vtype5, &zlen, &Z, nullptr, nullptr );
		assert( status == napi_ok );
		if ( vtype5 != napi_float64_array ) {
			napi_throw_type_error( env, nullptr, "invalid argument. Sixth argument must be a Float64Array." );
			return nullptr;
		}
		if ( (N-1)*llabs(strideZ) >= (int64_t)zlen ) {
			napi_throw_range_error( env, nullptr, "invalid argument. Sixth argument has insufficient elements based on the associated stride and the number of indexed elements." );
			return nullptr;
		}

		// Add the arrays to our array of array pointers:
		arrays[ 0 ] = (uint8_t *)X;
		arrays[ 1 ] = (uint8_t *)Y;
		arrays[ 2 ] = (uint8_t *)Z;

		// Perform addition:
		stdlib_strided_dd_d( arrays, shape, strides, (void *)add );

		return nullptr;
	}

	napi_value Init( napi_env env, napi_value exports ) {
		napi_status status;
		napi_value fcn;
		status = napi_create_function( env, "exports", NAPI_AUTO_LENGTH, node_add, NULL, &fcn );
		assert( status == napi_ok );
		return fcn;
	}

	NAPI_MODULE( NODE_GYP_MODULE_NAME, Init )
} // end namespace addon_strided_add
