
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; MODULE      : math-adjust-en.scm
;; DESCRIPTION : adjustments for English speech based on heuristic training
;; COPYRIGHT   : (C) 2022  Joris van der Hoeven
;;
;; This software falls under the GNU general public license version 3 or later.
;; It comes WITHOUT ANY WARRANTY WHATSOEVER. For details, see the file LICENSE
;; in the root directory or <http://www.gnu.org/licenses/gpl-3.0.html>.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(texmacs-module (math math-adjust-en)
  (:use (math math-speech)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Disambiguation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define differential-letters
  (list "f" "g" "h" "s" "t" "u" "v" "w" "x" "y" "z"))

(define (english-5/phi)
  (if (stats-has-variant? "<phi>")
      (speech-insert-symbol "<phi>")
      (speech-insert-symbol "5")))

(define (english-d/b/p-* . l)
  (if (and (== (length l) 1) (in? (car l) differential-letters))
      (speech-best-combination (list "<mathd>" "d" "b" "p") l)
      (speech-best-combination (list "d" "b" "p") l)))

(define (english-*-d/b/p . l)
  ;; TODO: raise the priority of <mathd> in the presence of integrals
  (speech-best-combination l (list "d" "b" "p" "<mathd>")))

(define (english-and)
  (if (stats-prefer? "<wedge>" "n" :normal)
      (speech-insert-symbol "<wedge>")
      (speech-insert-symbol "n")))

(define (english-in)
  (if (stats-prefer? "n" "<in>" :strong)
      (speech-insert-symbol "n")
      (speech-insert-symbol "<in>")))

(define (english-to)
  (if (inside? 'rsub)
      (with-innermost t 'rsub
        (with prev (tree-ref t :previous)
          (if (and (tree? prev) (tree-is? prev 'big))
              (speech-until)
              (speech-insert-number "2"))))
      (speech-insert-number "2")))

(define (english-for)
  (with prev (expr-before-cursor)
    (if (tm-is? prev 'big)
        (speech-for)
        (speech-insert-number "4"))))

(speech-map english math
  ;; a/e/8 ambiguity
  ("a/e/8" (speech-best-letter "a" "e" "8"))
  ("a/e/8 hat" (speech-best-accent "^" "a" "e"))
  ("a/e/8 tilda" (speech-best-accent "~" "a" "e"))
  ("a/e/8 bar" (speech-best-accent "<bar>" "a" "e"))

  ;; phi/5 ambiguity
  ("phi/5" (speech-best-letter "<phi>" "5"))
  ("5/phi" (english-5/phi))

  ;; t/d/v/3 and related ambiguities
  ("v/t/d/3" (speech-best-letter "v" "t" "d" "3"))
  ("d/t/v/3" (speech-best-letter "d" "t" "v" "3"))
  ("t/d/v/3" (speech-best-letter "t" "d" "v" "3"))
  ("t/d/v/3 hat" (speech-best-accent "^" "t" "d" "v"))
  ("t/d/v/3 tilda" (speech-best-accent "~" "t" "d" "v"))
  ("t/d/v/3 bar" (speech-best-accent "<bar>" "t" "d" "v"))
  ("d/t/v/3 u" (speech-insert-d "u"))
  ("d/t/v/3 v" (speech-insert-d "v"))
  ("d/t/v/3 w" (speech-insert-d "w"))
  ("d/t/v/3 x" (speech-insert-d "x"))
  ("d/t/v/3 y" (speech-insert-d "y"))
  ("d/t/v/3 z" (speech-insert-d "z"))

  ;; b/p/d ambiguity
  ("b/p" (speech-insert-best "b" "p"))
  ("b/p/d" (speech-insert-best "b" "p" "d"))
  ("b/p q" (speech-best-combination (list "b" "p") (list "q")))
  ("b/p t" (speech-best-combination (list "b" "p") (list "t")))

  ;; d/b/p/v and related ambiguities
  ("d u" (speech-insert-d "u"))
  ("d v" (speech-insert-d "v"))
  ("d w" (speech-insert-d "w"))
  ("d x" (speech-insert-d "x"))
  ("d y" (speech-insert-d "y"))
  ("d z" (speech-insert-d "z"))
  ("b/v" (speech-insert-best "b" "v"))
  ("d/b" (speech-insert-best "d" "b"))
  ("d/b/p" (speech-insert-best "d" "b" "p"))
  ("d/b/p b/p/d" (english-d/b/p-* "b" "p" "d"))
  ("d/b/p d/b" (english-d/b/p-* "d" "b"))
  ("d/b/p d/b/p" (english-d/b/p-* "d" "b" "p"))
  ("d/b/p d/v" (english-d/b/p-* "d" "v"))
  ("d/b/p s/f" (english-d/b/p-* "s" "f"))
  ("d/v" (speech-insert-best "d" "v"))
  ("d/v b/p/d" (speech-best-combination (list "d" "v") (list "b" "p" "d")))
  ("d/v d/b" (speech-best-combination (list "d" "v") (list "d" "b")))
  ("d/v d/b/p" (speech-best-combination (list "d" "v") (list "d" "b" "p")))
  ("d/v d/v" (speech-best-combination (list "d" "v") (list "d" "v")))
  ("d/v m/n" (speech-best-combination (list "d" "v") (list "m" "n")))
  ("d/v s/f" (speech-best-combination (list "d" "v") (list "s" "f")))
  
  ;; m/n and related ambiguities
  ("m/n" (speech-insert-best "m" "n"))
  ("m/n b/p/d" (speech-best-combination (list "m" "n") (list "b" "p" "d")))
  ("m/n d/b" (speech-best-combination (list "m" "n") (list "d" "b")))
  ("m/n d/b/p" (speech-best-combination (list "m" "n") (list "d" "b" "p")))
  ("m/n d/v" (speech-best-combination (list "m" "n") (list "d" "v")))
  ("m/n m/n" (speech-best-combination (list "m" "n") (list "m" "n")))
  ("m/n s/f" (speech-best-combination (list "m" "n") (list "s" "f")))

  ;; s/f and related ambiguities
  ("s/f" (speech-insert-best "s" "f"))
  ("s/f b/p/d" (speech-best-combination (list "s" "f") (list "b" "p" "d")))
  ("s/f d/b" (speech-best-combination (list "s" "f") (list "d" "b")))
  ("s/f d/b/p" (speech-best-combination (list "s" "f") (list "d" "b" "p")))
  ("s/f d/v" (speech-best-combination (list "s" "f") (list "d" "v")))
  ("s/f m/n" (speech-best-combination (list "s" "f") (list "m" "n")))
  ("s/f s/f" (speech-best-combination (list "s" "f") (list "s" "f")))

  ;; Other ambiguities
  ("j/g" (speech-insert-best "j" "g"))
  ("l/i" (speech-insert-best "l" "i"))
  ("xi/psi" (speech-insert-best "<xi>" "<psi>"))
  ("psi/xi" (speech-insert-best "<psi>" "<xi>"))
  ("in" (english-in))
  ("and" (english-and))
  ("to" (english-to))
  ("for" (english-for))
  )

(speech-map-wildcard english math
  ("d/b/p *" (english-d/b/p-* "*"))
  ("* d/b/p" (english-*-d/b/p "*"))
  ("m/n *" (speech-best-combination (list "m" "n") (list "*")))
  ("* m/n" (speech-best-combination (list "*") (list "m" "n")))
  ("s/f *" (speech-best-combination (list "s" "f") (list "*")))
  ("* s/f" (speech-best-combination (list "*") (list "s" "f")))
  ("* z/v" (speech-best-combination (list "*") (list "z" "v")))
  )

(speech-reduce english math
  ("plus eight" "plus a/e/8")
  ("minus eight" "minus a/e/8")
  ("times eight" "times a/e/8")
  ("eight plus" "a/e/8 plus")
  ("eight minus" "a/e/8 minus")
  ("eight times" "a/e/8 times")
  ("eight hat" "a/e/8 hat")
  ("eight tilda" "a/e/8 tilda")
  ("eight bar" "a/e/8 bar")

  ("bold three" "bold v/t/d/3")
  ("times three" "times d/t/v/3")
  ("three hat" "t/d/v/3 hat")
  ("three tilda" "t/d/v/3 tilda")
  ("three bar" "t/d/v/3 bar")

  ("greek five" "phi")
  ("plus five" "plus 5/phi")
  ("minus five" "minus 5/phi")
  ("times five" "times phi/5")
  ("five plus" "5/phi plus")
  ("five minus" "5/phi minus")
  ("five times" "5/phi times")
  ("big five" "big phi")
  ("bold five" "bold phi/5")
  ("upright five" "upright phi")
  ("five hat" "phi hat")
  ("five tilda" "phi tilda")
  ("five bar" "phi bar")

  ("the" "d/v")
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Tables to adjust recognition of mathematics inside text
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(speech-collection dangerous english
  ;; explicit commands
  "to" "factor" "power" "set" "element"

  ;; digits
  "too"

  ;; latin letters
  "a" "be" "sea" "seat" "see" "de" "day" "the"
  "he" "eat" "each" "if" "age" "edge"
  "i" "eye" "eyes" "either" "iron" "ok" "cake" "care" "case" "all" "old"
  "an" "and" "piece" "queue" "cute"
  "are" "art" "our" "yes" "chease" "tea" "team"
  "via" "you" "vegan" "ask" "eggs" "thanks" "why"

  ;; greek letters
  "grandma" "theater" "yoga" "copper" "laptop"
  "mute" "mood" "no" "new" "gnu" "knew" "site" "bye" "pie" "pipe"
  "road" "row" "role" "roll" "ciao" "towel" "tall" "toe" "toll" "town"
  "fight" "find" "fine" "fly" "sigh" "side" "size" "kind" "sky"

  ;; letter combinations
  "ecu" "easy" "bi" "busy" "agency" "icy" "ma" "empty" "auntie" "envy"
  "annex" "pity" "peezy" "cutie" "essay" "usually" "excess" "whitey"

  ;; variants
  "pick" "plastic"
  "both" "bowl" "build" "bouquet"
  "tractor"

  ;; binary operators and relations
  "class" "does" "play" "blessed" "please" "press" "minors"
  "time" "dancer"

  ;; textual operators
  "cosign" "lock" "luck" "look" "unlock" "timeslot"

  ;; brackets
  "bill" "seal" "cool" "tower" "final" "offbeat" "fan" "offend"
  "far" "office" "coffee" "affects" "effects" "fax" "fedex" "fix"
  "facts" "opposite" "offsite"

  ;; punctuation, brackets, big operators
  "dutch" "ducks" "of" "off" "some"

  ;; fractions, subscripts, superscripts
  "offer" "oversee" "overall" "overview"
  "saturday" "bishop" "sake" "subversion" "subway" "pizza" "visa"

  ;; wide accents
  "white" "head" "had" "hit" "hunt" "hurt" "pet" "cuba"

  ;; miscellaneous
  "it's" "write"

  ;; dangerous adjustments
  "by" "my" "sign" "end")

(speech-collection dangerous english
  "is")

(speech-collection skip-start english
  "and" "in" "or"
  "the" "integer" "integers" "rational" "rationals" "real" "reals"
  "sum" "product" "integral"
  "quotient" "remainder")

(speech-collection skip-end english
  "are" "as" "be" "he" "if" "in" "me" "or" "we"
  "quotient" "remainder")

(speech-collection skip english
  "and" "such" "that")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Adjust wrongly recognized words
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(speech-adjust english math
  ;; Adjust digits
  ("too" "two")
  ("digit to" "digit two")
  
  ;; Adjust latin letters
  ("hey" "a")
  ("be" "b")
  ("bea" "b")
  ("bee" "b")
  ("sea" "c")
  ("seat" "c")
  ("see" "c")
  ("siri" "c")
  ("day" "d")
  ("de" "d")
  ("dee" "d")
  ("he" "e")
  ("eat" "e")
  ("each" "e")
  ("if" "f")
  ("gee" "g")
  ("ji" "g")
  ("age" "h")
  ("edge" "h")
  ("hi" "i")
  ("eye" "i")
  ("eyes" "i")
  ("either" "i")
  ("i've" "i")
  ("iron" "i")
  ("jade" "j")
  ("jay" "j")
  ("ok" "k")
  ("kate" "k")
  ("katie" "k")
  ("cake" "k")
  ("care" "k")
  ("case" "k")
  ("que" "k")
  ("al" "l")
  ("el" "l")
  ("hell" "l")
  ("i'll" "l/i")
  ("em" "m/n")
  ("an" "n")
  ("oh" "o")
  ("all" "o")
  ("old" "o")
  ("00" "o")
  ("piece" "p")
  ("pee" "p")
  ("pete" "p")
  ("queue" "q")
  ("cues" "q")
  ("cute" "q")
  ("are" "r")
  ("art" "r")
  ("our" "r")
  ("cute" "q")
  ("yes" "s/f")
  ("ass" "s")
  ("cheese" "t")
  ("tea" "t")
  ("team" "t")
  ("tee" "t")
  ("you" "u")
  ("vegan" "v")
  ("via" "v")
  ("va" "v")
  ("ve" "v")
  ("a c t s/f" "x")
  ("acts" "x")
  ("ask" "x")
  ("ex" "x")
  ("eggs" "x")
  ("thanks" "x")
  ("why" "y")
  ("at work" "edward")
  ("hairy" "harry")
  ("isak" "isaac")
  
  ;; Adjust greek letters
  ("l5" "alpha")
  ("b that" "beta")
  ("beata" "beta")
  ("dita" "beta")
  ("peta" "beta")
  ("bita" "beta")
  ("rita" "beta")
  ("vita" "beta")
  ("beat a" "beta")
  ("beat that" "beta")
  ("gonna" "gamma")
  ("got my" "gamma")
  ("grandma" "gamma")
  ("gum all" "gamma")
  ("gum out" "gamma")
  ("gummer" "gamma")
  ("add simon" "epsilon")
  ("add cylon" "epsilon")
  ("abseiling" "epsilon")
  ("atta" "eta")
  ("eat a" "eta")
  ("eater" "eta")
  ("cheetah" "theta")
  ("tita" "theta")
  ("cita" "theta")
  ("theater" "theta")
  ("tata" "theta")
  ("santa" "theta")
  ("sciota" "iota")
  ("utah" "iota")
  ("yodel" "iota")
  ("yoga" "iota")
  ("yorktown" "iota")
  ("yota" "iota")
  ("capa" "kappa")
  ("capac" "kappa")
  ("capela" "kappa")
  ("carpet" "kappa")
  ("copper" "kappa")
  ("coppa" "kappa")
  ("cuppa" "kappa")
  ("kepa" "kappa")
  ("k-pop" "kappa")
  ("scopa" "kappa")
  ("lanta" "lambda")
  ("lampa" "lambda")
  ("lamp off" "lambda")
  ("lamp of" "lambda")
  ("lamp that" "lambda")
  ("lampke" "lambda")
  ("lampton" "lambda")
  ("laptop" "lambda")
  ("length of" "lambda")
  ("llamada" "lambda")
  ("locked up" "lambda")
  ("mantha" "lambda")
  ("manta" "lambda")
  ("mamta" "lambda")
  ("mew" "mu")
  ("mute" "mu")
  ("moo" "mu")
  ("mood" "mu")
  ("no" "nu")
  ("new" "nu")
  ("gnu" "nu")
  ("knew" "nu")
  ("site" "xi/psi")
  ("sigh" "xi/psi")
  ("all my chrome" "omicron")
  ("o n my chrome" "omicron")
  ("o my crumb" "omicron")
  ("oh my chrome" "omicron")
  ("omi chrome" "omicron")
  ("omi cron" "omicron")
  ("bye" "pi")
  ("pie" "pi")
  ("pipe" "pi")
  ("b r o" "rho")
  ("road" "rho")
  ("row" "rho")
  ("role" "rho")
  ("roll" "rho")
  ("set mac" "sigma")
  ("set my" "sigma")
  ("sick my" "sigma")
  ("suck my" "sigma")
  ("sync my" "sigma")
  ("ciao" "tau")
  ("towel" "tau")
  ("tall" "tau")
  ("tao" "tau")
  ("toe" "tau")
  ("toll" "tau")
  ("town" "tau")
  ("absolem" "upsilon")
  ("absalom" "upsilon")
  ("absalon" "upsilon")
  ("upside down" "upsilon")
  ("up salon" "upsilon")
  ("up so long" "upsilon")
  ("u p so long" "upsilon")
  ("f l y" "phi")
  ("fight" "phi")
  ("file" "phi")
  ("find" "phi")
  ("fine" "phi")
  ("fire" "phi")
  ("fly" "phi")
  ("side" "psi/xi")
  ("size" "psi/xi")
  ("psy" "psi")
  ("upside" "psi")
  ("kai" "chi")
  ("kite" "chi")
  ("kind" "chi")
  ("sky" "chi")
  ("amiga" "omega")
  ("oh my god" "omega")

  ;; Adjust letter combinations
  ("ah" "a h")
  ("ai" "a i")
  ("aij" "a i j")
  ("ecu" "a q")
  ("a.m." "a m/n")
  ("ar" "a r")
  ("easy" "a z")
  ("be be" "b b")
  ("be my" "b i")
  ("bi" "b i")
  ("bia" "b i")
  ("bij" "b i j")
  ("b q" "b/p q")
  ("b t" "b/p t")
  ("b b v" "b v")
  ("busy" "b z")
  ("b z" "b z/v")
  ("cij" "c i j")
  ("decu" "d/b q")
  ("d/b/p m/n v" "d/b/p v")
  ("dez" "d z")
  ("ea" "e a")
  ("e.g." "e g")
  ("ei" "e i")
  ("en" "e n")
  ("eo" "e o")
  ("eu" "e u")
  ("eax" "e x")
  ("exo" "e x")
  ("eyi" "e y")
  ("eli" "e y")
  ("iwai" "e y")
  ("is z" "e z")
  ("fo" "f o")
  ("fav" "f v")
  ("fve" "f v")
  ("fyi" "f y")
  ("f z" "f z/v")
  ("g.i." "g i")
  ("g-eazy" "g z")
  ("hiv" "h v")
  ("hve" "h v")
  ("hy" "h y")
  ("agency" "h z")
  ("ia" "i a")
  ("i'll be" "i b")
  ("icy" "i c")
  ("i.e." "i e")
  ("iah" "i h")
  ("i/o" "i o")
  ("ize" "i z")
  ("jo" "j o")
  ("jav" "j v")
  ("javy" "j v")
  ("ok baby" "k b")
  ("katie" "k t")
  ("ky" "k y")
  ("casey" "k z/v")
  ("k z" "k z/v")
  ("alby" "l b")
  ("aldi" "l d")
  ("lo" "l o")
  ("elzhi" "l z")
  ("ma" "m a")
  ("am i" "m i")
  ("empty" "m/n t")
  ("n b a" "n b")
  ("nah" "n h")
  ("n f l" "n l")
  ("n m/n" "n m")
  ("auntie" "n t")
  ("envy" "n v")
  ("annex" "n x")
  ("obi" "o b")
  ("oe" "o e")
  ("ol" "o l")
  ("ovi" "o v")
  ("o wi-fi" "o v")
  ("o z" "o z/v")
  ("pedi" "p d")
  ("p d/b/p" "p d/b")
  ("pe" "p e")
  ("pei" "p i")
  ("pii" "p i")
  ("pique" "p k")
  ("p.m." "p m/n")
  ("po" "p o")
  ("pikachu" "p q")
  ("petey" "p t")
  ("pity" "p t")
  ("pey" "p y")
  ("peezy" "p z")
  ("cutie" "q t")
  ("q v c" "q v")
  ("cousy" "q z")
  ("arcade" "r k")
  ("essay" "s a")
  ("se" "s e")
  ("ask you" "s q")
  ("sve" "s/f v")
  ("sy" "s y")
  ("s/f z a" "s/f z")
  ("ua" "u a")
  ("usa" "u a")
  ("you ate" "u a")
  ("u g g" "u g")
  ("ui" "u i")
  ("bluejay" "u j")
  ("u gay" "u j")
  ("uga" "u j")
  ("u owe" "u o")
  ("usee" "u z")
  ("usually" "u z")
  ("vijay" "v j")
  ("vo" "v o")
  ("voo" "v o")
  ("ves" "v s")
  ("vesey" "v z")
  ("wo" "w o")
  ("wor" "w r")
  ("next see" "x c")
  ("text c" "x c")
  ("xij" "x i j")
  ("text g" "x j/g")
  ("text jay" "x j/g")
  ("excess" "x s")
  ("yij" "y i j")
  ("why a m/n" "y m/n")
  ("my queue" "y q")
  ("my ass" "y s")
  ("my tea" "y t")
  ("whitey" "y t")
  ("why t v" "y t")
  ("wifey" "y v")
  ("zij" "z i j")
  ("zer" "z r")
  ("ziti" "z t")
  ("buy ice" "pi i")
  ("by i" "pi i")
  ("by iu" "pi i")
  ("bye bye" "pi i")
  ("bye-bye" "pi i")
  ("bye i" "pi i")
  ("to fight i" "two pi i")
  ("to buy i" "two pi i")

  ;; Adjust letter combinations involving Greek letters
  ("al-fayed" "alpha n")

  ;; Adjust capitalized letters
  ("dick" "big")
  ("pick" "big")
  ("beck" "big")
  ("plastic" "plus big")
  ("dixie" "big c")
  ("digby" "big b/p/d")
  ("make f" "big f")
  ("did g" "big g")
  ("bigeye" "big i")
  ("big guy" "big i")
  ("biko" "big o")
  ("waco" "big o")
  ("biggby" "big b/p/d")
  ("pick hour" "big r")
  ("make u" "big u")
  ("bequita" "big eta")
  ("bixi" "big xi")
  ("big side" "big xi")
  ("big site" "big xi")
  ("big five" "big phi")
  ("becky" "big chi")

  ;; Adjust bold variants
  ("bol" "bold")
  ("bolt" "bold")
  ("both" "bold")
  ("bowl" "bold")
  ("build" "bold")
  ("that's bold" "plus bold")
  ("bobby" "bold b")
  ("bowlby" "bold b")
  ("baldy" "bold d")
  ("boke" "bold k")
  ("bouquet" "bold k")
  ("volpe" "bold p")
  ("baltar" "bold r")
  ("bote" "bold t")
  ("bold wi‑fi" "bold v")
  ("bold fee" "bold v")
  ("bold feet" "bold v")
  ("volví" "bold v")
  ("bullseye" "bold z")
  ("bold five" "bold phi")
  ("bullfight" "bold phi")

  ;; Adjust upright variants
  ("a bright" "upright")
  ("a pride" "upright")
  ("upright five" "upright phi")

  ;; Adjust blackboard bold variants
  ("blackboard bolt" "blackboard bold")
  ("blackboard bolte" "blackboard bold")
  ("blackboard both" "blackboard bold")
  ("blackboard bowl" "blackboard bold")
  ("blackboard bull" "blackboard bold")
  ("blackboard volt" "blackboard bold")
  ("backward bold" "blackboard bold")
  ("blackbird" "blackboard")
  ("reels" "reals")

  ;; Adjust fraktur variants
  ("fractura" "fraktur")
  ("fracture" "fraktur")
  ("tractor" "fraktur")
  ("track to her" "fraktur")
  ("wreck to" "fraktur")
  ("plus factor" "plus fraktur")
  ("fraktur version" "fraktur v")

  ;; Adjust calligraphic variants
  ("carrie graphic" "calligraphic")
  ("carry graphic" "calligraphic")
  ("kelly graphic" "calligraphic")
  ("kenny graphic" "calligraphic")
  ("kevin graphic" "calligraphic")
  ("scary graphic" "calligraphic")
  ("skelly graphic" "calligraphic")
  ("skinny graphic" "calligraphic")
  ("tabby graphic" "calligraphic")

  ;; Adjust sans serif variants
  ("salsarita" "sans serif")
  ("salsaritas" "sans serif")
  ("salsa refill" "sans serif")
  ("salsa wreath" "sans serif")
  ("san sarith" "sans serif")
  ("san siri" "sans serif")
  ("sans sarith" "sans serif")
  ("sarith" "sans serif")
  ("saucer a" "sans serif")
  ("saucery" "sans serif")
  ("saulsberry" "sans serif")
  ("sensory" "sans serif")
  ("size siri" "sans serif")
  ("sounds arif" "sans serif")
  ("some sarife" "sans serif")
  ("some siri" "sans serif")
  ("sorcery" "sans serif")
  ("sounds sarith" "sans serif")
  ("sounds to reefs" "sans serif")
  ("sounds to reach" "sans serif")
  ("sarife" "sans serif")
  ("sunseri" "sans serif")
  ("seven siri" "sans serif")
  ("sans serif day" "sans serif d")
  ("sans serif femme" "sans serif m")
  ("sans serif 50" "sans serif t")
  ("sans serif vive" "sans serif v")
  ("sans serif v r" "sans serif v")
  ("sans serif v data" "sans serif beta")
  ("sans serif side" "sans serif psi")
  
  ;; Adjust 'letter plus' and 'plus letter'
  ("class" "plus")
  ("does" "plus")
  ("play" "plus")
  ("blessed" "plus")
  ("please" "plus")
  ("press" "plus")
  ("plusa" "plus a")
  ("iplus" "i plus")
  ("ipads" "i plus")
  ("buy plus" "pi plus")
  ("find plus" "phi plus")
  ("busqué" "plus k")
  ("plus speed" "plus p")
  ("plus a row" "plus rho")
  ("plus find" "plus phi")
  
  ;; Adjust 'letter minus' and 'minus letter'
  ("midas" "minus")
  ("minas" "minus")
  ("mind is" "minus")
  ("mine is" "minus")
  ("mines" "minus")
  ("minors" "minus")
  ("i'm minus" "i minus")
  ("rota minus" "iota minus")
  ("by minus" "pi minus")
  ("tell minus" "tau minus")
  ("find minus" "phi minus")
  ("guy minus" "chi minus")
  ("minus vive" "minus v")
  ("minus move" "minus mu")
  ("minus say" "minus xi")
  ("minus side" "minus xi")
  ("minus stahl" "minus tau")
  ("minus find" "minus phi")

  ;; Adjust 'letter times' and 'times letter'
  ("time" "times")
  ("and times" "n times")
  ("endtimes" "n times")
  ("play times" "p times")
  ("teetimes" "t times")
  ("let the times" "lambda times")
  ("buy times" "pi times")
  ("tell times" "tau times")
  ("tower times" "tau times")
  ("find times" "phi times")
  ("times a day" "times d")
  ("times day" "times d")
  ("times speed" "times p")
  ("times vive" "times v")
  ("times we" "times v")
  ("times better" "times beta")
  ("times stahl" "times tau")
  ("times start" "times tau")
  ("times find" "times phi")

  ;; Adjust basic relations
  ("ecua" "equal")
  ("equus" "equal")
  ("vehicles" "v equal")
  ("say equal" "xi/psi equal")
  ("buy equal" "pi equal")
  ("tell equal" "tau equal")
  ("equal samantha" "equal lambda")
  ("equal by" "equal pi")
  ("defined ass" "defined as")
  ("and less" "n less")
  ("and greater" "n greater")
  ("does not belong to" "not in")

  ;; Adjust further binary relations
  ("crossed" "cross")
  ("10 sir" "tensor")
  ("dancer" "tensor")

  ;; Adjust operator names
  ("cosign" "cosine")
  ("lock" "log")
  ("luck" "log")
  ("look" "log")
  ("clock" "log")
  ("log and" "log n")
  ("log in" "log n")
  ("unlock" "n log")
  ("timeslot" "times log")
  ("mott" "mod")
  ("modelo" "modulo")
  ("reminder" "remainder")

  ;; Adjust punctuation
  ("docs" "dots")
  ("dutch" "dots")
  ("ducks" "dots")
  ("call him" "colon")
  ("colin" "colon")
  ("column" "colon")
  ("call ma" "comma")
  ("call matt" "comma")
  ("call mark" "comma")
  ("call mom" "comma")
  ("call my" "comma")
  ("come out" "comma")
  ("come my" "comma")
  ("acoma" "a comma")
  ("ako my" "a comma")
  ("because my" "b comma")
  ("become a" "b comma")
  ("because i" "b comma")
  ("because i'm icy" "b comma c")
  ("biko massey" "b comma c")
  ("siri call maddie" "c comma d")
  ("decoma" "d comma")
  ("deco my" "d comma")
  ("eco my" "e comma")
  ("eco my ass" "e comma s/f")
  ("echo my" "e comma")
  ("yeah comma" "e comma")
  ("edgecomb" "h comma")
  ("i got my" "i comma")
  ("jacoby" "j comma b")
  ("jayco mckay" "j comma k")
  ("alcoma" "l comma")
  ("emco martin" "m comma n")
  ("i'm comin" "m comma n")
  ("how comma" "o comma")
  ("pico my" "p comma")
  ("plus comma" "p comma")
  ("cucumber" "q comma")
  ("arco my" "r comma")
  ("arkoma" "r comma")
  ("asko matty" "s comma t")
  ("esco matty" "s comma t")
  ("escrow matty" "s comma t")
  ("teco mavi" "t comma")
  ("tico made" "t comma")
  ("vico my" "v comma")
  ("w h y comma" "y comma")
  ("zico my" "z comma")
  ("vitico my" "beta comma")
  ("yo taco macapa" "iota comma kappa")
  ("call paco my" "kappa comma")
  ("capac o n m y" "kappa comma")
  ("capac o n my" "kappa comma")
  ("capac on my" "kappa comma")
  ("new commack" "nu comma")
  ("psycho my" "psi/xi comma")
  ("say comma" "psi/xi comma")
  ("set machel montano" "sigma comma tau")
  ("sick machel montano" "sigma comma tau")
  ("tell comma" "tau comma")
  ("fico maps" "phi comma")
  ("find comma" "phi comma")
  ("find commack" "phi comma")
  ("call my baby" "comma b")
  ("comedy" "comma d/b")
  ("call maggie" "comma g")
  ("comadre" "comma g")
  ("comma p a" "comma p")
  ("call marty" "comma t")
  ("comma v t" "comma v")
  ("kamasi" "comma z")
  ("comma doubt" "comma tau")
  
  ;; Adjust brackets
  ("off" "of")
  ("of of" "of")
  ("eight of" "a/e of")
  ("ayo fax" "a of x")
  ("be your" "b of")
  ("bill" "b of")
  ("seal" "c of")
  ("see your" "c of")
  ("dior" "d of")
  ("e f x" "e of x")
  ("eoff" "e of")
  ("f f" "f of")
  ("fof" "f of")
  ("fof" "f of")
  ("s/f f a" "f of a")
  ("s/f f b" "f of b")
  ("jaya" "j of")
  ("jayo" "j of")
  ("can you" "k of")
  ("lol" "l of")
  ("emma" "m of")
  ("i am a" "m of") 
  ("i'm of" "m of")
  ("eno" "n of")
  ("i know" "n of")
  ("n f x" "n of x")
  ("n f y" "n of y")
  ("nox" "n of x")
  ("p f x" "p of x")
  ("pio fax" "p of x")
  ("cool" "q of")
  ("haro" "r of")
  ("as of" "s of")
  ("ass of" "s of")
  ("s/f f x" "s of x")
  ("s/f f y" "s of y")
  ("three of" "t/d/v/3 of")
  ("theo" "t of")
  ("tio" "t of")
  ("u r of" "u of")
  ("v f x" "v of x")
  ("v f y" "v of y")
  ("version of" "v of")
  ("video of" "v of")
  ("my of" "y of")
  ("y r of" "y of")
  ("zio" "z of")
  ("alfalfa" "alpha of")
  ("how far of" "alpha of")
  ("beetle" "beta of")
  ("daltile" "delta of")
  ("zito" "zeta of")
  ("it'll" "eta of")
  ("ctown" "theta of")
  ("tito" "theta of")
  ("call paul" "kappa of")
  ("move of" "mu of")
  ("pi of" "pi of")
  ("set mouth" "sigma of")
  ("sick mouth" "sigma of")
  ("tower" "tau of")
  ("filofax" "phi of x")
  ("final" "phi of")
  ("find f" "phi of")
  ("five of" "phi of")
  ("scio" "psi of")
  ("kyle" "chi of")
  ("offbeat" "of b")
  ("of the" "of b/v")
  ("fan" "of n")
  ("offend" "of n")
  ("far" "of r")
  ("office" "of s")
  ("coffee" "of v")
  ("offie" "of v")
  ("affects" "of x")
  ("a fax" "of x")
  ("effects" "of x")
  ("fax" "of x")
  ("fedex" "of x")
  ("fix" "of x")
  ("i'll fax" "of x")
  ("facts" "of x")
  ("o fax" "of x")
  ("officemax" "of x")
  ("or fax" "of x")
  ("off-white" "of y")
  ("opposite" "of z")
  ("of data" "of beta")
  ("evita" "of eta")
  ("of m/n ø" "of mu")
  ("of by" "of pi")
  ("off-road" "of rho")
  ("05" "of 5/phi")
  ("of five" "of 5/phi")
  ("offsite" "of psi")
  ("rackets" "brackets")

  ;; Adjust big operators
  ("sam" "sum")
  ("some" "sum")
  ("sum four" "sum for")
  ("product four" "sum for")
  ("enter girl" "integral")
  ("enter grow" "integral")
  ("contours" "contour")
  ("come to her" "contour")
  ("come to our" "contour")
  ("ikon tour" "i contour")

  ;; Adjust subscripts
  ("sab" "sub")
  ("sup" "sub")
  ("sub script" "subscript")
  ("super script" "superscript")
  ("subscripted" "subscript")
  ("subscription" "subscript")
  ("subscription to" "subscript")
  ("subscripts" "subscript")
  ("subscribe" "subscript")
  ("subscribe to" "subscript")
  ("subscribed" "subscript")
  ("subscribed to" "subscript")
  ("sabe" "sub a")
  ("subsea" "sub c")
  ("saturday" "sub d")
  ("sera d" "sub d")
  ("sub g e" "sub g")
  ("subject" "sub g")
  ("sake" "sub k")
  ("subkey" "sub k")
  ("sub and" "sub n")
  ("sub end" "sub n")
  ("sub in" "sub n")
  ("sabol" "sub o")
  ("syrupy" "sub p")
  ("sub t v" "sub t")
  ("subversion" "sub v")
  ("subway" "sub y")
  ("sabzi" "sub z")
  ("subzero" "sub z")
  ("sabeta" "sub beta")
  ("asap" "a sub")
  ("bishop" "b sub")
  ("this up" "d sub")
  ("aesop" "e sub")
  ("isa" "i sub")
  ("i suck" "i sub")
  ("geez up" "g sub")
  ("case of" "k sub")
  ("quesadilla" "k sub d")
  ("elsa" "l sub")
  ("anseth" "n sub")
  ("pizza" "p sub")
  ("cues of" "q sub")
  ("cute up" "q sub")
  ("cusack" "q sub")
  ("tsop" "t sub")
  ("tsopk" "t sub k")
  ("visa" "v sub")
  ("my sub" "y sub")
  ("why is a" "y sub")
  ("zisa" "z sub")
  ("by sub" "pi sub")

  ;; Adjust powers and superscripts
  ("powerbeats" "power b")
  ("power and" "power n")
  ("power in" "power n")
  ("powers e" "power z")
  ("to power" "two power")
  ("empower" "m power")
  ("in power" "n power")
  ("is power" "e power")
  ("is superscript" "e superscript")
  ("exponent and" "exponent n")
  ("exponent in" "exponent n")
  ("superscript and" "superscript n")
  ("superscript in" "superscript n")
  ("e xquire" "x square")
  ("exquire" "x square")
  ("by square" "pi square")

  ;; Adjust fractions
  ("o for" "over")
  ("o from" "over")
  ("offer" "over")
  ("ayo for" "a over")
  ("do for" "d over")
  ("eo for" "e over")
  ("fo for" "f over")
  ("geo for" "g over")
  ("ho for" "h over")
  ("eye-opener" "i over")
  ("hi oh for" "i over")
  ("iovu" "i over")
  ("i'll old for" "i over")
  ("jo for" "j over")
  ("ko for" "k over")
  ("lo for" "l over")
  ("i'm over" "m over")
  ("eno for" "n over")
  ("po for" "p over")
  ("so for" "s over")
  ("to for" "t over")
  ("vo for" "v over")
  ("voo for" "v over")
  ("exo for" "x over")
  ("roll over" "rho over")
  ("find over" "phi over")
  ("fine over" "phi over")
  ("fios for" "phi over")
  ("overbay" "over b")
  ("overbee" "over b")
  ("overby" "over b")
  ("oversee" "over c")
  ("overall" "over o")
  ("overview" "over v")
  ("overtone" "over tau")

  ;; Adjust hat accents
  ("white" "wide")
  ("head" "hat")
  ("had" "hat")
  ("hit" "hat")
  ("hunt" "hat")
  ("hurt" "hat")
  ("pet" "hat")
  ("light head" "wide hat")
  ("why hat" "wide hat")
  ("why tilda" "wide tilda")
  ("why tilde" "wide tilda")
  ("why bar" "wide bar")
  ("my hat" "wide hat")
  ("my tilda" "wide tilda")
  ("my tilde" "wide tilda")
  ("my bar" "wide bar")
  ("he had" "e hat")
  ("jihad" "g hat")
  ("ipad" "i hat")
  ("in hat" "n hat")
  ("and hat" "n hat")
  ("asshat" "s hat")
  ("you had" "u hat")
  ("we hat" "v hat")
  ("we had" "v hat")
  ("cupperhead" "kappa hat")
  ("rohat" "rho hat")
  ("find hat" "phi hat")
  ("sign hat" "psi hat")
  ("side hat" "psi hat")

  ;; Adjust tilda accents
  ("in tilda" "n tilda")
  ("and tilda" "n tilda")
  ("find tilda" "phi tilda")
  ("sign tilda" "psi tilda")
  ("side tilda" "psi tilda")

  ;; Adjust bar accents
  ("bieber" "b bar")
  ("ebar" "e bar")
  ("aybar" "i bar")
  ("al-bahr" "l bar")
  ("embar" "m bar")
  ("end bar" "n bar")
  ("in bar" "n bar")
  ("and bar" "n bar")
  ("p b r" "p bar")
  ("q b r" "q bar")
  ("cuba" "q bar")
  ("eubar" "u bar")
  ("rebar" "v bar")
  ("rober" "rho bar")
  ("robar" "rho bar")
  ("rovar" "rho bar")
  ("tovar" "tau bar")
  ("find bar" "phi bar")
  ("sign bar" "psi bar")
  ("side bar" "psi bar")

  ;; Miscellaneous
  ("it's" "is")
  ("isn't" "is not")
  ("doesn't" "does not")
  ("write" "right")
  ("leaf" "leave")
  ("leafs" "leave")
  ("leaves" "leave")
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Further, more dangerous adjustments
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(speech-adjust english math
  ("08" "o a")
  ("28" "2 a")
  ("38" "3 a")
  ("48" "4 a")
  ("58" "5 a")
  ("81" "a 1")
  ("82" "a 2")
  ("83" "a 3")
  ("84" "a 4")
  ("85" "a 5")
  ("eight a m/n" "a m/n")

  ("de de" "d/b/p d/b/p")
  ("by" "b/p/d y")
  ("my" "m y")

  ("plus 80" "plus e")
  ("power a power" "power e power")
  ("a power e" "e power e")

  ("and log" "n log")
  ("end" "and")
  ("times 10" "times n")

  ("40" "for all")
  ("408" "for all a")

  ("search that" "such that")
  ("sign" "sine")
  ("10 four" "one over")
  ("104" "one over")

  ("exercise" "x i")
  )
