import { MembershipQueue, ProvisioningApi, ProvisioningRequest, Rules } from "matrix-appservice-bridge";
import { IrcServer } from "../irc/IrcServer";
import { IrcUser } from "../models/IrcUser";
import { IrcBridge } from "../bridge/IrcBridge";
export interface ProvisionerConfig {
    enabled: boolean;
    widget?: boolean;
    requestTimeoutSeconds: number;
    rules?: Rules;
    roomLimit?: number;
    http?: {
        port: number;
        host?: string;
    };
    secret?: string;
    apiPrefix?: string;
    ratelimit?: boolean;
    openIdDisallowedIpRanges?: string[];
    openIdOverrides?: Record<string, string>;
}
interface StrictProvisionerConfig extends ProvisionerConfig {
    secret: string;
}
export declare class Provisioner extends ProvisioningApi {
    readonly ircBridge: IrcBridge;
    readonly membershipQueue: MembershipQueue;
    readonly config: StrictProvisionerConfig;
    private pendingRequests;
    constructor(ircBridge: IrcBridge, membershipQueue: MembershipQueue, config: StrictProvisionerConfig);
    start(): Promise<void>;
    /**
     * Create a request to be passed internally to the provisioner.
     * @param fnName The function name to be called.
     * @param userId The userId, if specific to a user.
     * @param body The body of the request.
     * @param params The url parameters of the request.
     * @returns A ProvisioningRequest object.
     */
    static createFakeRequest(fnName: string, userId?: string, params?: {
        [key: string]: string;
    }, body?: unknown): ProvisioningRequest;
    private updateBridgingState;
    private userHasProvisioningPower;
    /**
     * Do a series of checks before contacting an operator for permission to create
     *  a provisioned mapping. If the operator responds with 'yes' or 'y', the mapping
     *  is created.
     * The checks done are the following:
     *  - (Matrix) Check power level of user is high enough
     *  - (IRC) Check that op's nick is actually a channel op
     *  - (Matrix) check room state to prevent route looping: don't bridge the same
     *    room-channel pair
     *  - (Matrix) update room state m.room.brdiging
     */
    private authoriseProvisioning;
    private sendToUser;
    /**
     * Contact an operator, asking for authorisation for a mapping, and if they reply
     * 'yes' or 'y', create the mapping.
     */
    private createAuthorisedLink;
    private removeRequest;
    /**
     * Returns a pending request if it's promise isPending(), otherwise null
     */
    private getRequest;
    private setRequest;
    handlePm(server: IrcServer, fromUser: IrcUser, text: string): Promise<void>;
    /**
     * Get information that might be useful prior to calling requestLink
     * @returns An array of IRC chan op nicks
     */
    queryLink(req: ProvisioningRequest): Promise<{
        operators: string[];
    }>;
    /**
     * Get the list of currently network instances.
     */
    queryNetworks(): Promise<{
        servers: {
            network_id: string;
            bot_user_id: string;
            desc: string;
            icon: string | undefined;
            fields: {
                domain: string;
            };
        }[];
    }>;
    /**
     * Link an IRC channel to a matrix room ID.
     */
    requestLink(req: ProvisioningRequest): Promise<void>;
    doLink(req: ProvisioningRequest, server: IrcServer, ircChannel: string, key: string | undefined, roomId: string, userId: string): Promise<void>;
    /**
     * Unlink an IRC channel from a matrix room ID
     * @param ignorePermissions If true, permissions are ignored (e.g. for bridge admins).
     * Otherwise, the user needs to be a Moderator in the Matrix room.
     */
    unlink(req: ProvisioningRequest, ignorePermissions?: boolean): Promise<void>;
    /**
     * Force the bot to leave both sides of a provisioned mapping if there are no more mappings that
     * map either the channel or room. Force IRC clients to part the channel.
     */
    leaveIfUnprovisioned(req: ProvisioningRequest, roomId: string, server: IrcServer, ircChannel: string): Promise<void>;
    /**
     * Parts IRC clients who should no longer be in the channel as a result of the given mapping being
     * unlinked.
     */
    private partUnlinkedIrcClients;
    private leaveMatrixVirtuals;
    /**
     *  Cause the bot to leave the matrix room if there are no other channels being mapped to
     * this room
     */
    private leaveMatrixRoomIfUnprovisioned;
    /**
     * List all mappings currently provisioned with the given matrix_room_id
     */
    listings(req: ProvisioningRequest): Promise<(false | {
        matrix_room_id: string;
        remote_room_channel: unknown;
        remote_room_server: unknown;
    })[]>;
    private getLimits;
}
export {};
