"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationInstanceBuilder = void 0;

var _uuid = require("uuid");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class IntegrationInstanceBuilder {
  constructor(client) {
    _defineProperty(this, "client", void 0);

    this.client = client;
  }

  async build(integration, options) {
    const instance = integration.deepCheck().then(result => {
      if (!result) {
        return Promise.reject(new Error('Integration is not valid'));
      }
    }).then(() => integration.getAssets()).then(assets => this.remapIDs(assets.savedObjects)).then(assets => this.remapDataSource(assets, options.dataSource)).then(assets => this.postAssets(assets)).then(refs => this.buildInstance(integration, refs, options));
    return instance;
  }

  remapDataSource(assets, dataSource) {
    if (!dataSource) return assets;
    assets = assets.map(asset => {
      if (asset.type === 'index-pattern') {
        asset.attributes.title = dataSource;
      }

      return asset;
    });
    return assets;
  }

  remapIDs(assets) {
    const toRemap = assets.filter(asset => asset.id);
    const idMap = new Map();
    return toRemap.map(item => {
      if (!idMap.has(item.id)) {
        idMap.set(item.id, (0, _uuid.v4)());
      }

      item.id = idMap.get(item.id);

      for (let ref = 0; ref < item.references.length; ref++) {
        const refId = item.references[ref].id;

        if (!idMap.has(refId)) {
          idMap.set(refId, (0, _uuid.v4)());
        }

        item.references[ref].id = idMap.get(refId);
      }

      return item;
    });
  }

  async postAssets(assets) {
    try {
      const response = await this.client.bulkCreate(assets);
      const refs = response.saved_objects.map(obj => {
        var _obj$attributes;

        return {
          assetType: obj.type,
          assetId: obj.id,
          status: 'available',
          // Assuming a successfully created object is available
          isDefaultAsset: obj.type === 'dashboard',
          // Assuming for now that dashboards are default
          description: (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title
        };
      });
      return Promise.resolve(refs);
    } catch (err) {
      return Promise.reject(err);
    }
  }

  async buildInstance(integration, refs, options) {
    const config = await integration.getConfig();
    return Promise.resolve({
      name: options.name,
      templateName: config.name,
      dataSource: options.dataSource,
      creationDate: new Date().toISOString(),
      assets: refs
    });
  }

}

exports.IntegrationInstanceBuilder = IntegrationInstanceBuilder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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