/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeaderName;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpPipelineNextSyncPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.ExponentialBackoff;
import com.azure.core.http.policy.FixedDelay;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.http.policy.RetryOptions;
import com.azure.core.http.policy.RetryStrategy;
import com.azure.core.implementation.ImplUtils;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LoggingEventBuilder;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;
import reactor.core.Exceptions;
import reactor.core.publisher.Mono;

public class RetryPolicy
implements HttpPipelinePolicy {
    private static final ClientLogger LOGGER = new ClientLogger(RetryPolicy.class);
    private final RetryStrategy retryStrategy;
    private final HttpHeaderName retryAfterHeader;
    private final ChronoUnit retryAfterTimeUnit;

    public RetryPolicy() {
        this(new ExponentialBackoff(), null, null);
    }

    public RetryPolicy(String retryAfterHeader, ChronoUnit retryAfterTimeUnit) {
        this(new ExponentialBackoff(), retryAfterHeader, retryAfterTimeUnit);
    }

    public RetryPolicy(RetryStrategy retryStrategy, String retryAfterHeader, ChronoUnit retryAfterTimeUnit) {
        this.retryStrategy = Objects.requireNonNull(retryStrategy, "'retryStrategy' cannot be null.");
        this.retryAfterHeader = HttpHeaderName.fromString(retryAfterHeader);
        this.retryAfterTimeUnit = retryAfterTimeUnit;
        if (!CoreUtils.isNullOrEmpty(retryAfterHeader)) {
            Objects.requireNonNull(retryAfterTimeUnit, "'retryAfterTimeUnit' cannot be null.");
        }
    }

    public RetryPolicy(RetryStrategy retryStrategy) {
        this(retryStrategy, null, null);
    }

    public RetryPolicy(RetryOptions retryOptions) {
        this(RetryPolicy.getRetryStrategyFromOptions(retryOptions), null, null);
    }

    private static RetryStrategy getRetryStrategyFromOptions(RetryOptions retryOptions) {
        Objects.requireNonNull(retryOptions, "'retryOptions' cannot be null.");
        if (retryOptions.getExponentialBackoffOptions() != null) {
            return new ExponentialBackoff(retryOptions.getExponentialBackoffOptions());
        }
        if (retryOptions.getFixedDelayOptions() != null) {
            return new FixedDelay(retryOptions.getFixedDelayOptions());
        }
        throw new IllegalArgumentException("'retryOptions' didn't define any retry strategy options");
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        return this.attemptAsync(context, next, context.getHttpRequest(), 0, null);
    }

    @Override
    public HttpResponse processSync(HttpPipelineCallContext context, HttpPipelineNextSyncPolicy next) {
        return this.attemptSync(context, next, context.getHttpRequest(), 0, null);
    }

    private Mono<HttpResponse> attemptAsync(HttpPipelineCallContext context, HttpPipelineNextPolicy next, HttpRequest originalHttpRequest, int tryCount, List<Throwable> suppressed) {
        context.setHttpRequest(originalHttpRequest.copy());
        context.setData("requestRetryCount", tryCount + 1);
        return next.clone().process().flatMap(httpResponse -> {
            if (RetryPolicy.shouldRetry(this.retryStrategy, httpResponse, tryCount)) {
                Duration delayDuration = RetryPolicy.determineDelayDuration(httpResponse, tryCount, this.retryStrategy, this.retryAfterHeader, this.retryAfterTimeUnit);
                RetryPolicy.logRetry(tryCount, delayDuration);
                httpResponse.close();
                return this.attemptAsync(context, next, originalHttpRequest, tryCount + 1, suppressed).delaySubscription(delayDuration);
            }
            if (tryCount >= this.retryStrategy.getMaxRetries()) {
                RetryPolicy.logRetryExhausted(tryCount);
            }
            return Mono.just((Object)httpResponse);
        }).onErrorResume(Exception.class, err -> {
            if (RetryPolicy.shouldRetryException(this.retryStrategy, err, tryCount)) {
                RetryPolicy.logRetryWithError(LOGGER.atVerbose(), tryCount, "Error resume.", err);
                List suppressedLocal = suppressed == null ? new LinkedList() : suppressed;
                suppressedLocal.add(err);
                return this.attemptAsync(context, next, originalHttpRequest, tryCount + 1, suppressedLocal).delaySubscription(this.retryStrategy.calculateRetryDelay(tryCount));
            }
            RetryPolicy.logRetryWithError(LOGGER.atError(), tryCount, "Retry attempts have been exhausted.", err);
            if (suppressed != null) {
                suppressed.forEach(err::addSuppressed);
            }
            return Mono.error((Throwable)err);
        });
    }

    private HttpResponse attemptSync(HttpPipelineCallContext context, HttpPipelineNextSyncPolicy next, HttpRequest originalHttpRequest, int tryCount, List<Throwable> suppressed) {
        HttpResponse httpResponse;
        context.setHttpRequest(originalHttpRequest.copy());
        context.setData("requestRetryCount", tryCount + 1);
        try {
            httpResponse = next.clone().processSync();
        }
        catch (RuntimeException err) {
            if (RetryPolicy.shouldRetryException(this.retryStrategy, err, tryCount)) {
                RetryPolicy.logRetryWithError(LOGGER.atVerbose(), tryCount, "Error resume.", err);
                try {
                    Thread.sleep(this.retryStrategy.calculateRetryDelay(tryCount).toMillis());
                }
                catch (InterruptedException ie) {
                    throw LOGGER.logExceptionAsError(new RuntimeException(ie));
                }
                LinkedList<Throwable> suppressedLocal = suppressed == null ? new LinkedList<Throwable>() : suppressed;
                suppressedLocal.add(err);
                return this.attemptSync(context, next, originalHttpRequest, tryCount + 1, suppressedLocal);
            }
            RetryPolicy.logRetryWithError(LOGGER.atError(), tryCount, "Retry attempts have been exhausted.", err);
            if (suppressed != null) {
                suppressed.forEach(err::addSuppressed);
            }
            throw LOGGER.logExceptionAsError(err);
        }
        if (RetryPolicy.shouldRetry(this.retryStrategy, httpResponse, tryCount)) {
            Duration delayDuration = RetryPolicy.determineDelayDuration(httpResponse, tryCount, this.retryStrategy, this.retryAfterHeader, this.retryAfterTimeUnit);
            RetryPolicy.logRetry(tryCount, delayDuration);
            httpResponse.close();
            try {
                Thread.sleep(this.retryStrategy.calculateRetryDelay(tryCount).toMillis());
            }
            catch (InterruptedException ie) {
                throw LOGGER.logExceptionAsError(new RuntimeException(ie));
            }
            return this.attemptSync(context, next, originalHttpRequest, tryCount + 1, suppressed);
        }
        if (tryCount >= this.retryStrategy.getMaxRetries()) {
            RetryPolicy.logRetryExhausted(tryCount);
        }
        return httpResponse;
    }

    private static boolean shouldRetry(RetryStrategy retryStrategy, HttpResponse response, int tryCount) {
        return tryCount < retryStrategy.getMaxRetries() && retryStrategy.shouldRetry(response);
    }

    private static boolean shouldRetryException(RetryStrategy retryStrategy, Throwable throwable, int tryCount) {
        if (tryCount >= retryStrategy.getMaxRetries()) {
            return false;
        }
        for (Throwable causalThrowable = Exceptions.unwrap((Throwable)throwable); causalThrowable != null; causalThrowable = causalThrowable.getCause()) {
            if (!retryStrategy.shouldRetryException(causalThrowable)) continue;
            return true;
        }
        return false;
    }

    private static void logRetry(int tryCount, Duration delayDuration) {
        LOGGER.atVerbose().addKeyValue("tryCount", tryCount).addKeyValue("durationMs", delayDuration.toMillis()).log("Retrying.");
    }

    private static void logRetryExhausted(int tryCount) {
        LOGGER.atInfo().addKeyValue("tryCount", tryCount).log("Retry attempts have been exhausted.");
    }

    private static void logRetryWithError(LoggingEventBuilder loggingEventBuilder, int tryCount, String format, Throwable throwable) {
        loggingEventBuilder.addKeyValue("tryCount", tryCount).log(format, throwable);
    }

    static Duration determineDelayDuration(HttpResponse response, int tryCount, RetryStrategy retryStrategy, HttpHeaderName retryAfterHeader, ChronoUnit retryAfterTimeUnit) {
        if (retryAfterHeader == null) {
            return RetryPolicy.getWellKnownRetryDelay(response.getHeaders(), tryCount, retryStrategy, OffsetDateTime::now);
        }
        String retryHeaderValue = response.getHeaderValue(retryAfterHeader);
        if (CoreUtils.isNullOrEmpty(retryHeaderValue)) {
            return retryStrategy.calculateRetryDelay(tryCount);
        }
        return Duration.of(Integer.parseInt(retryHeaderValue), retryAfterTimeUnit);
    }

    static Duration getWellKnownRetryDelay(HttpHeaders responseHeaders, int tryCount, RetryStrategy retryStrategy, Supplier<OffsetDateTime> nowSupplier) {
        Duration retryDelay = ImplUtils.getRetryAfterFromHeaders(responseHeaders, nowSupplier);
        if (retryDelay != null) {
            return retryDelay;
        }
        return retryStrategy.calculateRetryDelay(tryCount);
    }
}

