-- SPDX-License-Identifier: GPL-3.0-or-later

local upg_dir = '/var/lib/knot-resolver/.upgrade-4-to-5'
local out = upg_dir..'/kresd.conf.net'
local sockets = {
	{ file='kresd.socket', kind='dns' },
	{ file='kresd-tls.socket', kind='tls' },
	{ file='kresd-doh.socket', kind='doh2' },
	{ file='kresd-webmgmt.socket', kind='webmgmt' },
}

-- globals
addr_port = {}
outfile = io.open(out, 'w')

if outfile == nil then
	-- this is technically an error, but upgrade script shouldn't fail in scriptlets
	os.exit(0)  -- make no changes and exit
end

outfile:write("-- Suggested network interface configuration\n")
outfile:write("-- See https://knot-resolver.readthedocs.io/en/stable/upgrading.html\n\n")
outfile:write("-- Please remove any unused or undesired interfaces and add them to\n")
outfile:write("-- /etc/knot-resolver/kresd.conf\n\n")

local function write_net_listen(addr, port, kind)
	-- make sure (addr, port) combination is unique
	for _, val in ipairs(addr_port) do
		if val.addr == addr and val.port == port then
			return
		end
	end

	table.insert(addr_port, { addr=addr, port=port })
	outfile:write(
		"net.listen('"..addr.."', "..tostring(port)..
		", { kind = '"..kind.."', freebind = true })\n")
end

local function convert(line, kind, ipv6only)
	local patterns = {
		'^[^=]+=(%d+%.%d+%.%d+%.%d+):(%d+)',  -- IPv4
		'^[^=]+=%[([^%]]+)%]:(%d+)',  -- IPv6
		'^[^=]+=(/.*)',  -- UNIX
	}

	-- Datagram is either implied (dns) or unsupported (tls/doh/webmgmt)
	if not line:match('^Listen.*Stream') then
		return
	end

	for _, pattern in ipairs(patterns) do
		local addr, port = line:match(pattern)
		if addr ~= nil then
			write_net_listen(addr, port, kind)
			if not ipv6only then
				if addr:match('^::$') then
					write_net_listen('0.0.0.0', port, kind)
				end
				if addr:match('^::1$') then
					write_net_listen('127.0.0.1', port, kind)
				end
			end
		end
	end
	return
end

for _, socket in pairs(sockets) do
	local ipv6only = false
	local ipv6only_f = io.open(upg_dir..'/'..socket.file..'.v6only', 'r')
	if ipv6only_f ~= nil then
		ipv6only = true
		io.close(ipv6only_f)
	end
	local sockinfo = io.open(upg_dir..'/'..socket.file, 'r')
	if sockinfo ~= nil then
		for line in sockinfo:lines() do
			convert(line, socket.kind, ipv6only)
		end
	end
end

outfile:write("\n")

io.close(outfile)
os.exit(0)
