/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Spicebird code.
 *
 * The Initial Developer of the Original Code is
 * Synovel Software Technologies
 * Portions created by the Initial Developer are Copyright (C) 2008
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Prasad Sunkari <prasad@synovel.com> (Original Author)
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "csTPConnectionManager.h"
#include "nsComponentManagerUtils.h"
#include "nsIArray.h"
#include "nsIMutableArray.h"
#include "nsMemory.h"
#include "nsStringGlue.h"
#include "csITelepathyCallbacks.h"
#include "csTelepathyCID.h"
#include "csITPConnection.h"

#include "telepathy-glib/util.h"
#include "telepathy-glib/enums.h"
#include "telepathy-glib/gtypes.h"

NS_IMPL_ISUPPORTS1(csTPConnectionManager, csITPConnectionManager)
NS_IMPL_ISUPPORTS1(csTPConnectionManagerParam, csITPConnectionManagerParam)
NS_IMPL_ISUPPORTS1(csTPConnectionManagerProtocol, csITPConnectionManagerProtocol)

// Deliberately included after the NS_IMPL_ISUPPORTS
#include "csTelepathyMacros.h"

csTPConnectionManager::csTPConnectionManager()
{
  m_BusDaemon = tp_dbus_daemon_new(tp_get_bus());
  m_ConnectionManager = NULL;

  CS_TELEPATHY_INIT_PROXY
}

csTPConnectionManager::~csTPConnectionManager()
{
  if (m_ConnectionManager)
    g_object_unref(m_ConnectionManager);
}

/*
 * Call back for the "got-info" signal from the connection manager
 * This signal is emitted when ever the source of the information for
 * this connection-manager changes.  
 *
 * The possible sources are: 
 *  INFO_SOURCE_NONE - The connection manager has no information about
 *                     supported protocols and their parameters.
 *  INFO_SOURCE_FILE - The source of the current information is the
 *                     ".manager" file.
 *  INFO_SOURCE_LIVE - The source of current information is the
 *                     connection manager itself.  The connection-manager
 *                     is either running or was running when this
 *                     information was aquired.
 *
 *  telepathy-glib automatically populates the TpConnectionManager structure
 *  when information about the connection-manager is available.
 */
static void
GotInfoSignalled(TpConnectionManager *cm,
                guint source,
                void *weak_object)
{
  // TODO: Handle the GotInfo Signal
}

NS_IMETHODIMP csTPConnectionManager::SetNativeConnectionManager(TpConnectionManager *native)
{
  if (m_ConnectionManager)
    return NS_ERROR_ALREADY_INITIALIZED;

  m_ConnectionManager = native;
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManager::GetNativeConnectionManager(TpConnectionManager **native)
{
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  *native = m_ConnectionManager;
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManager::Init(const nsACString & aName)
{
  if (m_ConnectionManager)
    return NS_ERROR_ALREADY_INITIALIZED;

  m_ConnectionManager = tp_connection_manager_new(m_BusDaemon, nsCString(aName).get(), NULL, NULL);
  if (!m_ConnectionManager)
    return NS_ERROR_FAILURE;

  g_signal_connect((void*)m_ConnectionManager, "got-info", G_CALLBACK(GotInfoSignalled), NULL);

  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManager::GetName(nsACString & aName)
{
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  aName.AssignLiteral(m_ConnectionManager->name);
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManager::GetProtocols(nsIArray * *aProtocols)
{
  NS_ENSURE_ARG_POINTER(aProtocols);

  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  if (m_ConnectionManager->info_source <= TP_CM_INFO_SOURCE_NONE)
    return NS_ERROR_NOT_AVAILABLE;

  TpConnectionManagerProtocol * const *protocols = 
    (TpConnectionManagerProtocol *const *)m_ConnectionManager->protocols;
  while (*protocols != NULL) {
    g_message("%s", (*protocols)->name);
    *protocols++;
  }

  /* 
   * TODO: Create the array of protocols and return them
   *       Currently use GetProtocolList to get the list of protocols
   */
  return NS_ERROR_NOT_IMPLEMENTED;
}

/*
 * Callback for getting the list of supported protocols.
 * When |error| is not NULL, the list of protocols is passed in
 * the NULL terminated array |protocols|
 */
static void
GotProtocolList(TpConnectionManager *cm, const char **protocols, 
                const GError *error, gpointer user_data, GObject *unused)
{
  csITPCStringListCB *callback = (csITPCStringListCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);
  
  char **name = (char **)(protocols);
  while (*name != NULL) 
  {
    callback->OnAddItem(nsCString(*name));
    name++;
  }

  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

/*
 * GetProtocolList checks if the connection manager is already instrospected
 * and uses any information that is already available.  If no information
 * is available, it initiates a DBus call to get the list of protocols
 */
NS_IMETHODIMP csTPConnectionManager::GetProtocolList(csITPCStringListCB *cb)
{
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  if (m_ConnectionManager->info_source > TP_CM_INFO_SOURCE_NONE) {
    // I already have the data, don't bother
    // getting it from DBUS again!
    const TpConnectionManagerProtocol * const *it;
    for (it = m_ConnectionManager->protocols; it != NULL && *it != NULL; it++) 
      cb->OnAddItem(nsCString((*it)->name));

    cb->OnItemsComplete();
  } else {

    NS_IF_ADDREF(cb);
    tp_cli_connection_manager_call_list_protocols(m_ConnectionManager, 
                                                  -1, GotProtocolList,
                                                  cb, NULL, NULL);
  }
  return NS_OK;
}

/*
 * Callback for getting the list of paramters.
 * When |error| is not NULL, the list of parameters is passed in |arr|. 
 */
static void
GotParamsList(TpConnectionManager *cm, const GPtrArray *arr, 
              const GError *error, gpointer user_data, GObject *unused)
{
  csITPInterfaceListCB *callback = (csITPInterfaceListCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  PRUint32 length = arr->len;
  for (PRUint32 i=0; i<length; i++) {
    GValue structure = { 0 };
    GValue *tmp;
    TpConnectionManagerParam *param = g_new0(TpConnectionManagerParam, 1);

    g_value_init(&structure, TP_STRUCT_TYPE_PARAM_SPEC);
    g_value_set_static_boxed(&structure, g_ptr_array_index(arr, i));

    if (dbus_g_type_struct_get(&structure,
                               0, &param->name, 1, &param->flags,
                               2, &param->dbus_signature, 3, &tmp, G_MAXUINT)) {

      if (param->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT) {
        g_value_init(&param->default_value, G_VALUE_TYPE(tmp));
        g_value_copy(tmp, &param->default_value);
      }
      g_value_unset(tmp);
      g_free(tmp);

      csTPConnectionManagerParam *raw = new csTPConnectionManagerParam(param);
      nsCOMPtr<csITPConnectionManagerParam> param = do_QueryInterface(raw);
      callback->OnAddItem(param);
    }
  }
  
  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

/*
 * GetParamsList checks if the connection manager is already instrospected
 * and uses any information that is already available.  If no information
 * is available, it initiates a DBus call to get the list of parameters.
 */
NS_IMETHODIMP csTPConnectionManager::GetParamsList(csITPInterfaceListCB *cb,
                                                   const nsACString & aProtocolName)
{
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;
  
  if (m_ConnectionManager->info_source > TP_CM_INFO_SOURCE_NONE) {

    const TpConnectionManagerProtocol * const *it;
    bool foundProtocol = false;
    for (it = m_ConnectionManager->protocols; it != NULL && *it != NULL; it++) {

      if (g_str_equal((*it)->name, nsCString(aProtocolName).get())) {

        const TpConnectionManagerParam *param = (*it)->params;
        foundProtocol = true;

        for (param = (*it)->params; param->name != NULL; param++) {
          csTPConnectionManagerParam *raw = new csTPConnectionManagerParam(param);
          nsCOMPtr<csITPConnectionManagerParam> param = do_QueryInterface(raw);
          cb->OnAddItem(param);
        }
        cb->OnItemsComplete();
      }
    }
    if (!foundProtocol)
      return NS_ERROR_INVALID_ARG;
  } else {
    NS_IF_ADDREF(cb);
    tp_cli_connection_manager_call_get_parameters(m_ConnectionManager, -1,
                                                  nsCString(aProtocolName).get(),
                                                  GotParamsList, cb, NULL, NULL);
  }

  return NS_OK;
}

/*
 * Callback for the RequestConnection telepathy call
 * When |error| is not NULL, the DBus bus name is passed in |busName|
 * and the object path in |objectPath|.
 *
 * Apart from this callback, whenever a new connection is established
 * a "new-connection" signal is also emitted by the connection manager.
 *
 * The signal is even more useful when the connections are established
 * by other applications - it helps in good Desktop integration!
 */
static void
GotRequestConnectionResponse(TpConnectionManager *cm, const gchar *serviceName,
                             const gchar *objectPath, const GError *error,
                             gpointer user_data, GObject *unused)
{
  csITPInterfaceCB *callback = (csITPInterfaceCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);
  
  nsCOMPtr<csITPConnection> connection = do_CreateInstance(CS_TPCONNECTION_CONTRACTID);
  connection->Init(nsDependentCString(serviceName), nsDependentCString(objectPath));

  callback->OnGotValue(connection);
  NS_IF_RELEASE(callback);
}

typedef struct {
  csITPInterfaceCB *connectionCB;
  csITPGetParamCB *paramsCB;
  nsCString protocolName;
  TpConnectionManager *manager;
} Get_Params_For_Connection;

static void
GotParamsForNewConnection(TpConnectionManager *cm, const GPtrArray *arr, 
                          const GError *error, gpointer user_data,
                          GObject *unused)
{
  Get_Params_For_Connection* cb_struct = (Get_Params_For_Connection*)user_data;
  csITPInterfaceCB *cb = cb_struct->connectionCB;
  csITPGetParamCB *paramsCB = cb_struct->paramsCB;
  nsCString protocolName = cb_struct->protocolName;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, cb, error);

  PRUint32 length = arr->len;
  GHashTable *hash = g_hash_table_new(g_str_hash, g_str_equal);
  nsresult rv;

  for (PRUint32 i=0; i<length; i++) {
    nsCOMPtr<nsIVariant> variant;
    GValue structure = { 0 };
    TpConnectionManagerParam *param = g_new0(TpConnectionManagerParam, 1);

    g_value_init(&structure, TP_STRUCT_TYPE_PARAM_SPEC);
    g_value_set_static_boxed(&structure, g_ptr_array_index(arr, i));

    if (dbus_g_type_struct_get(&structure,
                               0, &param->name, 1, &param->flags,
                               2, &param->dbus_signature, G_MAXUINT)) {
      // Check if its a secret param.  The implementations
      // might want to treat secret params differently.
      if (param->flags & TP_CONN_MGR_PARAM_FLAG_SECRET || 
          g_str_equal(param->name, "password")) {
        rv = paramsCB->GetSecretValueFor(nsCString(param->name),
                                         nsCString(param->dbus_signature),
                                         getter_AddRefs(variant));
      } else {
        rv = paramsCB->GetValueFor(nsCString(param->name), 
                                   nsCString(param->dbus_signature),
                                   getter_AddRefs(variant));
      }

      if (NS_SUCCEEDED(rv) && variant) {
        GValue *value = csTPConnectionManager::GetParamValue(param, variant);
        g_hash_table_insert(hash, g_strdup(param->name), value);
      }
    }
  }
  
  tp_cli_connection_manager_call_request_connection(cb_struct->manager, -1, 
                                                    protocolName.get(), hash, 
                                                    GotRequestConnectionResponse,
                                                    cb, NULL, NULL);
  NS_IF_RELEASE(paramsCB);
}

/*
 * Request a new connection.
 * This function retrieves the list of parameters supported by the
 * connection-manager and uses the |paramsCB| to get the values for
 * each of the parameters.  Prepares a GHashTable out of it and
 * initiates the connection request.
 */
NS_IMETHODIMP 
csTPConnectionManager::RequestConnection(csITPInterfaceCB* cb,
                                         csITPGetParamCB* paramsCB, 
                                         const nsACString & aProtocolName)
{
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  if (m_ConnectionManager->info_source > TP_CM_INFO_SOURCE_NONE) {

    GHashTable *hash = g_hash_table_new(g_str_hash, g_str_equal);
    const TpConnectionManagerProtocol * const *it;
    PRBool foundProtocol = PR_FALSE;

    for (it = m_ConnectionManager->protocols; it != NULL && *it != NULL; it++) {
      if (g_str_equal((*it)->name, nsCString(aProtocolName).get())) {
        const TpConnectionManagerParam *param = (*it)->params;
        nsresult rv;
        foundProtocol = PR_TRUE;

        for (param = (*it)->params; param->name != NULL; param++) {
          nsCOMPtr<nsIVariant> variant;

          // Check if its a secret param.  The implementations
          // might want to treat secret params differently.
          if (param->flags & TP_CONN_MGR_PARAM_FLAG_SECRET || 
              g_str_equal(param->name, "password")) {
            rv = paramsCB->GetSecretValueFor(nsCString(param->name),
                                             nsCString(param->dbus_signature),
                                             getter_AddRefs(variant));
          } else {
            rv = paramsCB->GetValueFor(nsCString(param->name), 
                                       nsCString(param->dbus_signature),
                                       getter_AddRefs(variant));
          }

          if (NS_SUCCEEDED(rv) && variant) {
            GValue *value = GetParamValue(param, variant);
            g_hash_table_insert(hash, g_strdup(param->name), value);
          }
        }

        NS_IF_ADDREF(cb);
        tp_cli_connection_manager_call_request_connection(m_ConnectionManager, -1, 
                                                          nsCString(aProtocolName).get(), hash, 
                                                          GotRequestConnectionResponse,
                                                          cb, NULL, NULL);
      }
    }
    if (foundProtocol == PR_FALSE && cb != NULL) {
      return NS_ERROR_INVALID_ARG;
    }
  } else {
    /* 
     * Get parameters using callback and
     * then request connection from that callback
     */

    NS_IF_ADDREF(cb);
    NS_IF_ADDREF(paramsCB);

    Get_Params_For_Connection *cb_struct = new Get_Params_For_Connection;
    cb_struct->connectionCB = cb;
    cb_struct->paramsCB = paramsCB;
    cb_struct->manager = m_ConnectionManager;
    cb_struct->protocolName = aProtocolName;

    tp_cli_connection_manager_call_get_parameters(m_ConnectionManager, -1,
                                                  nsCString(aProtocolName).get(),
                                                  GotParamsForNewConnection,
                                                  cb_struct, NULL, NULL);
  }
  return NS_OK;
}

/*
 * Convenience function to convert nsIVariant to GValue
 * based on the param signature
 */
GValue *
csTPConnectionManager::GetParamValue(const TpConnectionManagerParam *aParam, 
                                     nsIVariant *aVariantValue)
{
  GValue *value = g_new0(GValue, 1);
  nsCString cStrValue;
  nsresult rv;

  switch (aParam->dbus_signature[0]) {

    case 's':
      rv = aVariantValue->GetAsACString(cStrValue);
      if (NS_FAILED(rv))
        goto ValueError;

      g_value_init(value, G_TYPE_STRING);
      g_value_set_string(value, cStrValue.get());
      break;

    case 'b':
      PRBool boolValue;
      rv = aVariantValue->GetAsBool(&boolValue);
      if (NS_FAILED(rv))
        goto ValueError;

      g_value_init(value, G_TYPE_BOOLEAN);
      g_value_set_boolean(value, boolValue);
      break;

    case 'q':
    case 'u':
      PRUint32 uint32Value;
      rv = aVariantValue->GetAsUint32(&uint32Value);
      if (NS_FAILED(rv))
        goto ValueError;

      g_value_init(value, G_TYPE_UINT);
      g_value_set_uint(value, uint32Value);
      break;

    case 'n':
    case 'i':
      PRInt32 int32Value;
      rv = aVariantValue->GetAsInt32(&int32Value);
      if (NS_FAILED(rv))
        goto ValueError;

      g_value_init(value, G_TYPE_INT);
      g_value_set_int(value, int32Value);
      break;

    case 't':
      PRUint64 uint64Value;
      rv = aVariantValue->GetAsUint64(&uint64Value);
      if (NS_FAILED(rv))
        goto ValueError;

      g_value_init(value, G_TYPE_UINT64);
      g_value_set_uint64(value, uint64Value);
      break;

    case 'x':
      PRInt64 int64Value;
      rv = aVariantValue->GetAsInt64(&int64Value);
      if (NS_FAILED(rv))
        goto ValueError;

      g_value_init(value, G_TYPE_INT64);
      g_value_set_int64(value, int64Value);
      break;

    case 'd':
      double doubleValue;
      rv = aVariantValue->GetAsDouble(&doubleValue);
      if (NS_FAILED(rv))
        goto ValueError;

      g_value_init(value, G_TYPE_DOUBLE);
      g_value_set_double(value, doubleValue);
      break;

    default:
#ifdef _DEBUG
      g_print("Oops! Unknown Parameter Type!\n");
#endif
      break;
  }

  return value;

ValueError:
  g_free(value);
  return NULL;
}

NS_IMETHODIMP csTPConnectionManager::GetRunning(PRBool *aRunning)
{
  NS_ENSURE_ARG_POINTER(aRunning);
  
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  *aRunning = m_ConnectionManager->running;
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManager::GetInfoSource(PRUint32 *aInfoSource)
{
  NS_ENSURE_ARG_POINTER(aInfoSource);
  
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  *aInfoSource = m_ConnectionManager->info_source;
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManager::GetAlwaysIntrospect(PRBool *aAlwaysIntrospect)
{
  NS_ENSURE_ARG_POINTER(aAlwaysIntrospect);
  
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  *aAlwaysIntrospect = m_ConnectionManager->always_introspect;
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManager::SetAlwaysIntrospect(PRBool aAlwaysIntrospect)
{
  if (!m_ConnectionManager)
    return NS_ERROR_NOT_INITIALIZED;

  g_object_set(m_ConnectionManager, "always-introspect", aAlwaysIntrospect, NULL);
  return NS_OK;
}

/*
 * csITPConnectionManagerProtocol Implementation
 */

csTPConnectionManagerProtocol::csTPConnectionManagerProtocol()
{
}

csTPConnectionManagerProtocol::~csTPConnectionManagerProtocol()
{
}

void csTPConnectionManagerProtocol::Init(TpConnectionManagerProtocol *cmp)
{
}

NS_IMETHODIMP csTPConnectionManagerProtocol::GetName(nsACString & aName)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPConnectionManagerProtocol::GetParams(nsIArray * *aParams)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

/*
 * csITPConnectionManagerParam Implementation
 */

csTPConnectionManagerParam::csTPConnectionManagerParam()
{
  /*
   * This class is only constructed from inside the
   * csTPConnectionManager class where the other
   * constructor is called
   */
}

csTPConnectionManagerParam::csTPConnectionManagerParam(const TpConnectionManagerParam *param)
{
  m_ConnectionManagerParam = param;

  if (m_ConnectionManagerParam->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT) {

    nsCOMPtr<nsIWritableVariant> writable = do_CreateInstance(NS_VARIANT_CONTRACTID);
    GValue *value = (GValue *)(&(param->default_value));
    int type = G_VALUE_TYPE(value);

    /*
     * Please refer to dbus-gvalue.c in the dbus-glib 
     * source distribution for more information on these mappings.
     *
     * Based on version 0.76 of dbus-glib
     */
    switch(type) {

      case G_TYPE_UCHAR:
        writable->SetAsUint8(g_value_get_uchar(value));
        break;

      case G_TYPE_BOOLEAN:
        writable->SetAsBool(g_value_get_boolean(value));
        break;

      case G_TYPE_INT:
        writable->SetAsInt32(g_value_get_int(value));
        break;

      case G_TYPE_UINT:
        writable->SetAsUint32(g_value_get_uint(value));
        break;

      case G_TYPE_INT64:
        writable->SetAsInt64(g_value_get_int64(value));
        break;

      case G_TYPE_UINT64:
        writable->SetAsUint64(g_value_get_uint64(value));
        break;

      case G_TYPE_DOUBLE:
        writable->SetAsDouble(g_value_get_double(value));
        break;

      case G_TYPE_STRING:
        writable->SetAsString(g_value_get_string(value));
        break;

      default:
        g_message("Invalid default value type!");
    }

    m_DefaultValue = do_QueryInterface(writable);
  }
}

csTPConnectionManagerParam::~csTPConnectionManagerParam()
{
}

NS_IMETHODIMP csTPConnectionManagerParam::GetName(nsACString & aName)
{
  if (!m_ConnectionManagerParam)
    return NS_ERROR_NOT_INITIALIZED;

  aName.AssignLiteral(m_ConnectionManagerParam->name);
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManagerParam::GetSignature(nsACString & aSignature)
{
  if (!m_ConnectionManagerParam)
    return NS_ERROR_NOT_INITIALIZED;

  aSignature.AssignLiteral(m_ConnectionManagerParam->dbus_signature);
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManagerParam::GetFlags(PRUint32 *aFlags)
{
  if (!m_ConnectionManagerParam)
    return NS_ERROR_NOT_INITIALIZED;

  *aFlags = m_ConnectionManagerParam->flags;
  return NS_OK;
}

NS_IMETHODIMP csTPConnectionManagerParam::GetDefaultValue(nsIVariant **aProtocols)
{
  if (!m_ConnectionManagerParam)
    return NS_ERROR_NOT_INITIALIZED;

  if ((m_ConnectionManagerParam->flags & TP_CONN_MGR_PARAM_FLAG_HAS_DEFAULT) == 0)
    return NS_ERROR_NOT_AVAILABLE;

  NS_IF_ADDREF(*aProtocols = m_DefaultValue);

  return NS_OK;
}

CS_TELEPATHY_IMPL_PROXY(csTPConnectionManager, m_ConnectionManager);
