#!/bin/sh
#

set -e

if [ -z "$1" ] || [ -z "$2" ]; then
    echo "Missing parameter"
    echo "Usage: $0 src_version up_version"
    echo
    exit 1;
fi


version=$1
upversion=$2
hg_branch="qutecom-2.2"
curdir="$PWD"
destdir=qutecom-$version
desttar=qutecom_$version.orig.tar
uptar=qutecom_$upversion.orig.tar.gz

if [ -d "$curdir/../tarballs" ]; then
    finaldir=$curdir/../tarballs
else
    finaldir=$(dirname $curdir)
fi

if [ -e "$finaldir/$desttar.gz" ]; then
    echo "Uptodate archive found ($finaldir/$desttar.gz), nothing to do."
    exit 0
fi

workdir="$(mktemp -d --tmpdir=$(pwd))"
trap "rm -fr $workdir; echo Aborted." EXIT

# Download sources
if [ -d $workdir/$destdir ]; then
    echo "Uptodate source found ($destdir), skipping download."
    cd $workdir
else
    echo "Downloading sources... "
    if ( echo "$upversion" | grep -E 'hg[0-9]+' > /dev/null ); then
        hg_rev=$(echo "$upversion" | sed -e 's/^.*hg\([0-9]*\).*$/\1/')
        hg_filename="$hg_rev.tar.bz2"
        hg_dirname="$(echo -n $hg_branch | tr -s '.' '-')-$hg_rev"
        hg_url="http://hg.qutecom.org/$hg_branch/archive/$hg_filename"
        wget "$hg_url"
        tar jxf "$hg_filename"
        mv "$hg_dirname" "$destdir"
        rm "$hg_filename"
    else
        filename=$(basename $(uscan --no-symlink --destdir "." --force-download --download-version "$upversion" --dehs | sed -n 's/.*<upstream-url>\(.*\)<\/upstream-url>.*/\1/p'))
        cd $workdir
        tar zxf "../$filename"
        mv [Qq]ute[Cc]om* "$destdir"
        rm "../$filename"
    fi
    echo "done."
fi


# Cleaning tree
purge_3rdparty() {
    purgedir="$1"
    if [ -z "$purgedir" ] || [ ! -d "$purgedir" ]; then
        return;
    fi
    find $purgedir -depth -type f -not -iname cmake\* -print0 | xargs -0 rm -vf
    find $purgedir -depth -type d -not -iname cmake\* -print0 | xargs -0 rmdir --ignore-fail-on-non-empty
}

echo "Cleaning tree to remove anything not needed for non-Windows builds..."

# Remove windows binary stuff
rm -vrf $destdir/qutecom/nsis
rm -vrf $destdir/libs/3rdparty/*/binary-lib
find $destdir -iname \*.dll -print0 | xargs -0 rm -vf
find $destdir -iname \*.lib -print0 | xargs -0 rm -vf


# Remove outdated autotools files
find $destdir \( -iname config.guess -o -iname config.sub \) -print0 | xargs -0 rm -vf

# Remove iLBC codec
rm -rfv $destdir/wifo/phapi/ilbc

# Remove portaudio code
purge_3rdparty $destdir/libs/3rdparty/portaudio 

# Remove ffmpeg code
purge_3rdparty $destdir/libs/3rdparty/ffmpeg 
rm -r $destdir/libs/3rdparty/ffmpeg-bin-win32

# Remove speex code
purge_3rdparty $destdir/wifo/phapi/speex

# Remove samplerate code
purge_3rdparty $destdir/libs/3rdparty/samplerate

# Remove curl code
# note: libray needed for Debian etch and Ubuntu feisty
purge_3rdparty $destdir/libs/3rdparty/curl

# Remove iaxclient code
purge_3rdparty $destdir/libs/3rdparty/iaxclient

# Remove pthread code
purge_3rdparty $destdir/libs/3rdparty/pthread

# Remove glib code
purge_3rdparty $destdir/libs/3rdparty/glib

# Remove openssl code
purge_3rdparty $destdir/libs/3rdparty/openssl

# Remove libpurple code
purge_3rdparty $destdir/libs/3rdparty/libpurple

# Remove gnutls code
purge_3rdparty $destdir/libs/3rdparty/gnutls

# Remove libiconv code
purge_3rdparty $destdir/libs/3rdparty/iconv

# Remove libintl code
purge_3rdparty $destdir/libs/3rdparty/libintl

# Remove libxml code
purge_3rdparty $destdir/libs/3rdparty/libxml

# Remove zlib code
purge_3rdparty $destdir/libs/3rdparty/zlib

# Remove sipX code
purge_3rdparty $destdir/libs/sipx

# Remove MSVC code
purge_3rdparty $destdir/libs/owutil/cutil/msvc

# Remove MSVC reit libraries
purge_3rdparty $destdir/libs/3rdparty/libstdcpp

# Avoid lintian warning on source: ancient-libtool wifo/eXosip/scripts/ltmain.sh 1.4.3
rm -f $destdir/wifo/eXosip/scripts/ltmain.sh

# Avoid lintian warning on source: source-contains-hg-tags-file .hgtags
rm -f $destdir/.hg*

echo "Cleaning done"

# Make the sources archive
echo -n "Writing archive... "
tar cf $desttar $destdir
gzip --best $desttar
rm -r $destdir
echo "done"

mv -f $desttar.gz "$finaldir"
echo "Archive is in $finaldir/$desttar.gz"

rm -fr "$workdir"

trap "" EXIT

exit 0
